package ru.yandex.tours.backend.flight

import org.joda.time.{DateTime, DateTimeZone, Duration}
import ru.yandex.tours.avia.{Airports, AviaClient, FlightUtils}
import ru.yandex.tours.model.search.FlightSearchRequest
import ru.yandex.tours.model.search.SearchProducts.FlightTrip
import ru.yandex.tours.util.{Logging, Metrics}

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success}

// Simple class to query and process AviaSearch result not in an actor thread but synchronously (for Avia Cache APIs)
class SimpleAviaSearcher(aviaClient: AviaClient, airports: Airports)
                        (implicit ec: ExecutionContext) extends Logging {

  private val metrics = Metrics("avia.searcher")
  private val timer = metrics.getTimer("time")
  private val errors = metrics.getMeter("errors")
  private val found = metrics.getMeter("found")
  private val notFound = metrics.getMeter("not_found")
  private val allInsane = metrics.getMeter("all_insane")

  private def convertResponseVariants(variants: Seq[AviaClient.RaspFare],
                                      flights: Map[String, AviaClient.RaspFlight],
                                      companies: Map[Int, AviaClient.RaspCompany]): Seq[FlightTrip] = {
    val result = Vector.newBuilder[FlightTrip]
    val numInvalids = variants.count(!_.correct)
    if (numInvalids > 0) {
      log.warn(s"$numInvalids variants with invalid routes or currencies found")
    }
    val correctVars = variants.filter(_.correct)
    val ord = Ordering.by((_: AviaClient.RaspFare).totalPrice)
    val minVars = correctVars.groupBy(v => (v.forward, v.backward)).
      map(el => el._1 -> el._2.reduceOption(ord.min))

    for (((forward, backward), v) <- minVars if (forward.isDefined || backward.isDefined) && v.isDefined) {
      val builder = FlightTrip.newBuilder()
      builder.setMinPrice(v.get.totalPrice)
      if (forward.isDefined) {
        fillFlight(flights(forward.get), builder.addForwardBuilder(), companies)
      }
      if (backward.isDefined) {
        fillFlight(flights(backward.get), builder.addBackwardBuilder(), companies)
      }
      result += builder.build()
    }
    result.result()
  }

  private def fillFlight(flight: AviaClient.RaspFlight,
                         builder: FlightTrip.Flight.Builder,
                         companies: Map[Int, AviaClient.RaspCompany]): Unit = {
    val departure = DateTime.parse(flight.departure.local).withZoneRetainFields(DateTimeZone.forID(flight.departure.tzname))
    val arrival = DateTime.parse(flight.arrival.local).withZoneRetainFields(DateTimeZone.forID(flight.arrival.tzname))
    val duration = new Duration(departure, arrival).getStandardMinutes.toInt

    builder
      .setFlightNumber(flight.number)
      .setDeparture("s" + flight.from.toString)
      .setArrival("s" + flight.to.toString)
      .setDepartureTime(departure.toString)
      .setArrivalTime(arrival.toString)
      .setDuration(duration)
      .setCompany(companies(flight.company).title)
  }

  private def isValidFlightTrip(request: FlightSearchRequest)(flightTrip: FlightTrip): Boolean = {
    def check(flights: Seq[FlightTrip.Flight]): Boolean = {
      for (flight <- flights) {
        if (FlightUtils.getFlightDuration(flight, flight).getStandardMinutes <= 0) {
          log.warn(s"Negative flight duration for $request \n $flight")
          return false
        }
        if (airports.byAviaId(flight.getDeparture).isEmpty) {
          log.warn(s"Unknown departure airport id for $request: ${flight.getDeparture}")
          return false
        }
        if (airports.byAviaId(flight.getArrival).isEmpty) {
          log.warn(s"Unknown arrival airport id for $request: ${flight.getArrival}")
          return false
        }
      }
      for (Seq(first, second) <- flights.sliding(2)) {
        if (FlightUtils.getTransitionDuration(first, second).getStandardMinutes <= 0) {
          log.warn(s"Negative transition time for $request \n $first \n-----\n $second")
          return false
        }
      }
      true
    }
    check(flightTrip.getForwardList.asScala) && check(flightTrip.getBackwardList.asScala)
  }

  private def collectMetrics(request: FlightSearchRequest)(valid: Seq[FlightTrip],
                                                           insaneFlights: Seq[FlightTrip]): Unit = {
    if (valid.nonEmpty) found.mark() else notFound.mark()
    if (insaneFlights.nonEmpty) {
      if (valid.isEmpty) allInsane.mark()
      log.warn(s"Found ${insaneFlights.size} insane flights for request $request")
    }
  }

  def getFlights(request: FlightSearchRequest): Future[Seq[FlightTrip]] = {
    log.info(s"Starting simple avia search for request $request")
    val context = timer.time()
    val f = aviaClient.getMinPriceFromRasp(request)
    var res = for (resp <- f) yield {
      context.stop()
      val flightMap = resp.flights.map(f => f.key -> f).toMap
      val compMap = resp.companies.map(c => c.id -> c).toMap
      val response = convertResponseVariants(resp.fares, flightMap, compMap)
      val (valid, insaneFlights) = response.partition(isValidFlightTrip(request))
      collectMetrics(request)(valid, insaneFlights)
      valid
    }
    res.recover {
      case t: Throwable =>
        log.error(s"Can not get avia results for $request", t)
        context.stop()
        errors.mark()
        Seq.empty
    }
  }
}
