package ru.yandex.tours.backend.recommend

import java.time.Month

import org.joda.time.LocalDate
import ru.yandex.tours.direction.Direction
import ru.yandex.tours.geo.Departures
import ru.yandex.tours.model.Prices.{DirectionBestPrice, Recommendation, RecommendedDirection}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{SearchType, WhereToGoRequest}
import ru.yandex.tours.personalization.{DirectionInterestService, UserIdentifiers}
import ru.yandex.tours.search.DefaultRequestGenerator._
import ru.yandex.tours.storage.direction.DirectionPriceStorage
import ru.yandex.tours.util.Randoms._

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}


class DirectionRecommendationService(directionInterest: DirectionInterestService,
                                     toursPriceStorage: DirectionPriceStorage,
                                     roomsPriceStorage: DirectionPriceStorage,
                                     departures: Departures,
                                     searchType: SearchType)
                                    (implicit ec: ExecutionContext) {

  private val MONTHS_TO_SCAN = 3

  def getRecommendation(request: Option[WhereToGoRequest],
                        directions: Seq[Direction],
                        userIdentifiers: UserIdentifiers,
                        geoId: Int): Future[Recommendation] = {
    val preferredDirections = directionInterest.getDirections(userIdentifiers)
    val pricesFuture = getDirectionPrices(request, directions, geoId)
    for {
      preferred <- preferredDirections
      prices <- pricesFuture
    } yield {
      Recommendation.newBuilder()
        .addAllPreferredDirections(preferred.map(Int.box).asJava)
        .addAllDirections(prices.asJava)
        .build
    }
  }

  private def getDirectionPrices(request: Option[WhereToGoRequest],
                                 directions: Seq[Direction],
                                 geoId: Int): Future[Seq[RecommendedDirection]] = {
    Future.sequence(
      for {
        direction <- directions
        directionMonths = getMonths(request, direction)
        if directionMonths.nonEmpty
        budget = request.flatMap(_.budget)
      } yield {
        getDirectionWithPrice(departures.getDepartures(geoId, direction.region.id), direction, directionMonths, budget)
      }
    )
  }

  private def getDirectionWithPrice(fromOrder: Seq[Int],
                                    direction: Direction,
                                    months: Iterable[Month],
                                    budget: Option[Int]): Future[RecommendedDirection] = {
    require(fromOrder.nonEmpty)
    val firstFrom = fromOrder.head
    val firstPrice = getDirectionPrice(firstFrom, direction, months, budget)
    fromOrder.tail.foldLeft(firstPrice) {
      case (future, nextFrom) =>
        future.flatMap { result =>
          if (result.hasPrice) Future.successful(result)
          else getDirectionPrice(nextFrom, direction, months, budget)
        }
    }.flatMap { result =>
      if (result.hasPrice) Future.successful(result)
      else firstPrice
    }
  }

  private def getDirectionPrice(from: Int, direction: Direction,
                                months: Iterable[Month],
                                budget: Option[Int]): Future[RecommendedDirection] = {
    val tours = toursPriceStorage.getPrices(from, DEFAULT_AGES, direction.region.id, months.toSet)
    val rooms = roomsPriceStorage.getPrices(from, DEFAULT_AGES, direction.region.id, months.toSet)
    for {
      tours <- tours
      rooms <- rooms
    } yield {
      val tourPrices = getMinPricesForNights(tours.flatMap(_.prices))
      val roomPrices = getMinPricesForNights(rooms.flatMap(_.prices))
      val contextPrices = if (searchType == SearchType.ROOMS) roomPrices else tourPrices
      val (effectivePrices, effectiveContext) =
        if (tourPrices.nonEmpty) tourPrices -> SearchType.TOURS
        else roomPrices -> SearchType.ROOMS

      val defaultRequest = getDefaultRequest(from, direction.region.id, months.head, searchType)
      val builder = RecommendedDirection.newBuilder()
        .setGeoId(direction.region.id)
        .setOBSOLETEImage(direction.images.randomElement.toProto)
        .setImage(direction.images.randomElement.toProto)
        .setSquareImage(direction.squareImages.randomElement.toProto)
        .setHasCard(direction.hasCard)
        .setSearchRequest(defaultRequest.toProto)

      if (contextPrices.nonEmpty) { // todo remove after new snippets (HOTELS-1663)
        val (_, bestPrice) = getBestPrice(contextPrices, searchType, budget)
        builder.setSearchRequest(bestPrice.getSearchRequest)
        builder.setPrice(bestPrice.getPrice)
      }

      if (effectivePrices.nonEmpty) {
        val (bestNights, _) = getBestPrice(effectivePrices, effectiveContext, budget)
        tourPrices.get(bestNights).foreach(builder.setTourPrice)
        roomPrices.get(bestNights).foreach(builder.setRoomPrice)
      }
      builder.build
    }
  }

  private def getBestPrice(prices: Map[Int, DirectionBestPrice],
                           context: SearchType, budget: Option[Int]): (Int, DirectionBestPrice) = {
    budget match {
      case Some(b) =>
        prices.minBy { case (nights, price) => (price.getPrice - b).abs }
      case None =>
        if (context == SearchType.ROOMS) {
          prices.minBy { case (nights, price) => (price.getPrice, -nights) }
        } else {
          val (long, short) = prices.partition(_._1 > 5)
          if (long.nonEmpty) long.minBy(_._2.getPrice)
          else short.minBy { case (nights, price) => price.getPrice / nights }
        }
    }
  }

  private def getMinPricesForNights(prices: Seq[DirectionBestPrice]): Map[Int, DirectionBestPrice] = {
    prices.groupBy(_.getSearchRequest.getNights).map {
      case (nights, p) => nights -> p.minBy(_.getPrice)
    }
  }

  private def getDefaultMonths = {
    val currentMonth = Month.of(LocalDate.now.getMonthOfYear)
    ((0 until MONTHS_TO_SCAN) map (currentMonth.plus(_))).toSet
  }

  private def getMonths(request: WhereToGoRequest): Set[Month] = {
    request.month.fold(getDefaultMonths)(Set(_))
  }

  private def getMonths(request: Option[WhereToGoRequest], direction: Direction): Set[Month] = {
    request.fold(getDefaultMonths) { req => getMonths(req).filter(direction.seasonFor(req.thematic).contains) }
  }
}
