package ru.yandex.tours.backend.recommend

import ru.yandex.tours.direction.{Direction, Directions, DirectionsStats}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.Prices.{Recommendation, RecommendedDirection}
import ru.yandex.tours.model.direction.Thematics._
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.WhereToGoRequest
import ru.yandex.tours.model.util.SortType
import ru.yandex.tours.model.util.SortType.SortType
import ru.yandex.tours.personalization.UserIdentifiers
import ru.yandex.tours.services.RecommendService
import ru.yandex.tours.storage.direction.DirectionPriceStorage
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.Dates._

import scala.collection.JavaConversions._
import scala.concurrent.{ExecutionContext, Future}

/* @author berkut@yandex-team.ru */

class LocalRecommendService(directionRecommender: DirectionRecommendationService,
                            directions: Directions,
                            directionStats: DirectionsStats,
                            directionPriceStorage: DirectionPriceStorage,
                            tree: Tree,
                            searchType: SearchType)
                           (implicit ec: ExecutionContext) extends RecommendService with Logging {

  private val MAX_SHOWCASE_DIRECTIONS = 1

  /** Returns directions recommendation for given user; used in showcase */
  override def recommendTopDirections(userIdentifiers: UserIdentifiers, userRegion: Int): Future[Recommendation] = {
    val (promoted, directions) = findDirections(userRegion, None)

    directionRecommender.getRecommendation(None, directions, userIdentifiers, userRegion).map { recommendation =>
      val interestIds = recommendation.getPreferredDirectionsList.toSet.map(Int.unbox)
      val ordering = orderingByPriority(userRegion, interestIds, promoted)
      val sorted = SortType.sort(SortType.RELEVANCE, recommendation.getDirectionsList, ordering)
      val sliced = sorted.take(MAX_SHOWCASE_DIRECTIONS)
      val marked = markPromotedDirections(sliced, promoted)
      recommendation.toBuilder
        .clearDirections()
        .addAllDirections(marked)
        .build()
    }
  }

  /** Returns directions recommendation for given user; used on landing page */
  override def recommendDirections(userIdentifiers: UserIdentifiers,
                                   userRegion: Int, sortBy: SortType): Future[Recommendation] = {
    val (promoted, directions) = findDirections(userRegion, None)
    directionRecommender.getRecommendation(None, directions, userIdentifiers, userRegion).map { recommendation =>
      val interestIds = recommendation.getPreferredDirectionsList.toSet.map(Int.unbox)
      val ordering = orderingByPriority(userRegion, interestIds, promoted)
      val sorted = SortType.sort(sortBy, recommendation.getDirectionsList, ordering)
      val sliced = sorted.drop(MAX_SHOWCASE_DIRECTIONS)
      val marked = markPromotedDirections(sliced, promoted)
      recommendation.toBuilder
        .clearDirections()
        .addAllDirections(marked)
        .build()
    }
  }

  override def recommendAllDirections(userIdentifiers: UserIdentifiers,
                                      userRegion: Int, sortBy: SortType,
                                      branding: Option[String]): Future[Recommendation] = {
    val (promoted, directions) = findDirections(userRegion, branding)
    log.debug("Starting fetching  " + directions.size.toString + " directions")
    val started = System.currentTimeMillis()
    directionRecommender.getRecommendation(None, directions, userIdentifiers, userRegion).map { recommendation =>
      log.debug("Prices fetched in " + (System.currentTimeMillis() - started))
      val interestIds = recommendation.getPreferredDirectionsList.toSet.map(Int.unbox)
      val ordering = orderingByPriority(userRegion, interestIds, promoted)
      val sorted = SortType.sort(sortBy, recommendation.getDirectionsList, ordering)
      val marked = markPromotedDirections(sorted, promoted)
      val res = recommendation.toBuilder
        .clearDirections()
        .addAllDirections(marked)
        .build()
      res
    }
  }

  /** Returns directions recommendation for given user and country */
  override def recommendResorts(countryId: Int,
                                userIdentifiers: UserIdentifiers,
                                userRegion: Int): Future[Recommendation] = {
    val (promoted, directions) = findDirections(userRegion, None)
    val resorts = directions.filter(d => tree.country(d.region).exists(_.id == countryId))

    directionRecommender.getRecommendation(None, resorts, userIdentifiers, userRegion).map { recommendation =>
      val interestIds = recommendation.getPreferredDirectionsList.toSet.map(Int.unbox)
      val ordering = orderingByPriority(userRegion, interestIds, promoted)
      val sorted = SortType.sort(SortType.RELEVANCE, recommendation.getDirectionsList, ordering)
      val marked = markPromotedDirections(sorted, promoted)
      recommendation.toBuilder
        .clearDirections()
        .addAllDirections(marked)
        .build()
    }
  }

  /** Returns directions recommendation for given user; used on thematic page */
  override def recommendDirections(request: WhereToGoRequest, userIdentifiers: UserIdentifiers,
                                   userRegion: Int, sortBy: SortType): Future[Recommendation] = {
    val directions = findDirections(request, userRegion)
    val future = directionRecommender.getRecommendation(Some(request), directions, userIdentifiers, userRegion)
    future.map { recommendation =>
      val preferredDirections = recommendation.getPreferredDirectionsList.toSet.map(Int.unbox)
      val ordering = orderingBy(request.thematic, request.budget, userRegion, preferredDirections)
      val sorted = SortType.sort(sortBy, recommendation.getDirectionsList, ordering)
      recommendation.toBuilder
        .clearDirections()
        .addAllDirections(sorted)
        .build()
    }
  }

  /** directions for main page */
  private def findDirections(userRegion: Int, branding: Option[String]): (Map[Int, String], Vector[Direction]) = {
    val promotedIds = branding match {
      case Some("jordan") => Map(10535 -> "jordan")
      case _ => Map.empty[Int, String]
    }
    val promoted = promotedIds.keys.flatMap(directions.get).toVector
    val resorts = directions.all
      .filterNot(_.isCountry)
      .filter(_.region.id != userRegion)
    promotedIds -> (promoted ++ resorts)
  }

  /** directions for thematic page */
  private def findDirections(request: WhereToGoRequest, userRegion: Int): Vector[Direction] = {
    val thematic = request.thematic
    var found = directions.all
      .filter(!_.isCountry)
      .filter(_.region.id != userRegion)
      .filter(_.relevance(thematic) > 0)

    if (request.noVisa) found = found.filter(_.noVisa)
    request.month.foreach { month =>
      found = found.filter(_.seasonFor(thematic).contains(month))
    }
    request.countryId.foreach { countryId =>
      found = found.filter(d => tree.pathToRoot(d.region).exists(_.id == countryId))
    }
    found
  }

  private def getPrices(dp: RecommendedDirection): Seq[Int] = {
    Seq(
      if (dp.hasTourPrice) Some(dp.getTourPrice.getPrice) else None,
      if (dp.hasRoomPrice && dp.getRoomPrice.hasFlightPrice) Some(dp.getRoomPrice.getPrice) else None
    ).flatten
  }

  private def getBudgetBoost(price: Int, budget: Int): Double = {
    val diff = ((budget - price).toDouble / budget).abs
    def withinBudget(threshold: Double) = diff <= threshold

    if (withinBudget(0.1d)) 500d
    else if (withinBudget(0.2d)) 400d
    else if (withinBudget(0.3d)) 300d
    else (30d / diff) min 100d
  }

  private def getLowPriceBoost(prices: Seq[Int]): Double = {
    if (prices.nonEmpty) {
      if (prices.min < 200000) 400d
      else if (prices.min < 100000) 500d
      else if (prices.min < 15000) 600d
      else 0d
    } else 0d
  }

  private def markPromotedDirections(directions: Seq[RecommendedDirection], brandings: Map[Int, String]) = {
    directions.map { direction =>
      brandings.get(direction.getGeoId) match {
        case Some(campaign) =>
          direction.toBuilder
            .setBrandingCampaign(campaign)
            .build()
        case None => direction
      }
    }
  }

  private def orderingBy(thematic: Thematic, budget: Option[Int], fromGeoId: Int,
                         interestIds: Set[Int]): Ordering[RecommendedDirection] = Ordering.by {
    dp =>
      val when = dp.getSearchRequest.getWhen.toLocalDate
      directions.get(dp.getGeoId).fold(0d) { d =>
        val withPriceBoost = if (dp.hasRoomPrice || dp.hasTourPrice) 1000d else 0d
        val prices = getPrices(dp)
        val withinBudgetBoost = if (prices.nonEmpty && budget.isDefined) getBudgetBoost(prices.min, budget.get) else 0d
        val sameDepartureBoost = if (dp.getSearchRequest.getFrom == fromGeoId) 50d else 0d
        val highSeasonBoost = if (d.highSeasonFor(thematic).contains(when)) 5d else 0d
        val interestBoost = if (tree.pathToRoot(dp.getGeoId).exists(d => interestIds.contains(d.id))) 3d else 0d
        val relevance = d.relevance(thematic)

        relevance + highSeasonBoost + withPriceBoost + withinBudgetBoost + sameDepartureBoost + interestBoost
      }
  }
  private def orderingByPriority(fromGeoId: Int, interestIds: Set[Int],
                                 brandingSet: Map[Int, String]): Ordering[RecommendedDirection] = Ordering.by {
    dp =>
      val when = dp.getSearchRequest.getWhen.toLocalDate
      val withPriceBoost = if (dp.hasRoomPrice || dp.hasTourPrice) 1000d else 0d
      val prices = getPrices(dp)
      val lowPriceBoost = getLowPriceBoost(prices)
      val sameDepartureBoost = if (dp.getSearchRequest.getFrom == fromGeoId) 50d else 0d

      val relevance = directions.get(dp.getGeoId).fold(0d)(_.relevance(when))
      val priority = directionStats.getPriority(dp.getGeoId, searchType)
      val normalizedPriority = (priority / 5d) + 1d // some magic

      val interestBoost = if (tree.pathToRoot(dp.getGeoId).exists(d => interestIds.contains(d.id))) 6d else 0d
      val promotedBoost = directions.get(dp.getGeoId).fold(0d) { d => if (d.isPromoted) 20d else 0d }
      val brandingBoost = if (brandingSet.contains(dp.getGeoId)) 2000d else 0d

      (normalizedPriority * relevance) + withPriceBoost + lowPriceBoost +
        sameDepartureBoost + promotedBoost + interestBoost + brandingBoost
  }
}
