package ru.yandex.tours.backend.search

import akka.actor.{ActorRef, ReceiveTimeout}
import org.joda.time.DateTime
import ru.yandex.tours.events.SearchEvent.Actualized
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.GetOfferRequest
import ru.yandex.tours.model.search.SearchResults.{ActualizedOffer, Context}
import ru.yandex.tours.partners.PartnerProtocol.{ActualizeTourOffer, TourOfferGot}
import ru.yandex.tours.storage.ToursDao

import scala.concurrent.Promise
import scala.concurrent.duration._
import scala.util.{Failure, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.04.15
 */
class ActualizeRequestHolder(storage: ToursDao,
                             request: GetOfferRequest,
                             tourOffer: ActualizedOffer,
                             promise: Promise[Option[ActualizedOffer]],
                             partners: Map[Partner, ActorRef],
                             timeout: Duration = 30.seconds)
  extends AbstractRequestHolder[GetOfferRequest, ActualizedOffer](storage, request, partners) {

  override def preStart(): Unit = {
    super.preStart()
    context.setReceiveTimeout(timeout)
  }

  override protected def getRequestToPartner(partner: Partner, ctx: Context): Option[AnyRef] = {
    if (tourOffer.getOffer.getSource.getPartnerId == partner.id) {
      Some(ActualizeTourOffer(request, tourOffer))
    } else Option.empty
  }


  override def receive: Receive = {
    case TourOfferGot(Success(actualized)) =>
      promise.success(Some(actualized))
      context.system.eventStream.publish(Actualized(DateTime.now, request, actualized))
      context.stop(self)
    case TourOfferGot(Failure(exception)) =>
      log.warn(s"Failed to actualize tour", exception)
      promise.success(None)
      context.stop(self)
    case ReceiveTimeout =>
      log.warn(s"Timeout on actualize tour")
      promise.success(None)
      context.stop(self)
  }
}
