package ru.yandex.tours.backend.search

import akka.actor.{ActorRef, Props}
import org.joda.time.DateTime
import ru.yandex.tours.backend.search.HotelRequestHolder.MAX_HOTELS_TO_SEARCH
import ru.yandex.tours.events.SearchEvent.FoundSnippets
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.filter.hotel.PartnerIdFilter
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.search.SearchProducts.HotelSnippet
import ru.yandex.tours.model.search.SearchResults.{Context, HotelSearchResult, ResultInfo}
import ru.yandex.tours.operators.{SearchSourceAvailability, SearchSources}
import ru.yandex.tours.partners.PartnerProtocol
import ru.yandex.tours.partners.PartnerProtocol._
import ru.yandex.tours.storage.ToursDao
import ru.yandex.tours.util.Collections._

import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.concurrent.Future
import scala.util.{Failure, Success}

class HotelRequestHolder[S <: Source](storage: ToursDao,
                                      request: HotelSearchRequest,
                                      hotelsIndex: HotelsIndex,
                                      tree: Tree,
                                      partners: Map[Partner, ActorRef],
                                      searchSources: SearchSources[S],
                                      searchSourceAvailability: SearchSourceAvailability[S])
  extends SnippetRequestHolder[HotelSearchRequest, HotelSearchResult, S, SnippetsResult](storage, request, partners,
    searchSources, searchSourceAvailability) {

  import context.dispatcher

  private var loggedFiltered = false

  override protected def metricsName: String = "hotels_search"

  override protected def saveResponse(res: HotelSearchResult): Future[Unit] =
    storage.saveHotelSearchResult(request, res)

  override protected def createSearchEvent(request: HotelSearchRequest, res: HotelSearchResult) =
    FoundSnippets(DateTime.now, request, res)

  private case class PrioritySnippet(snippet: HotelSnippet, priority: Int) {
    val key = (snippet.getHotelId, snippet.getWithFlight)
  }

  private val prioritySnippetOrdering = Ordering.by { snippet: PrioritySnippet =>
    (snippet.priority, -snippet.snippet.getOfferCount, snippet.snippet.getPriceMin)
  }

  override protected def createResult(results: Iterable[SnippetsResult]): HotelSearchResult = {
    val foundSnippets: Iterable[HotelSnippet] = results.groupBy(_.searchSource).flatMap {
      case (source, sourceResults) =>
        val sourceSnippets = sourceResults.flatMap {
          case SnippetsResult(_, Successful(snippets), _) => snippets.map(PrioritySnippet(_, 0))
          case SnippetsResult(_, Partial(snippets), _) => snippets.map(PrioritySnippet(_, 1))
          case _ => Iterable.empty
        }
        sourceSnippets.groupBy(_.key).map {
          case (_, similarSnippets) =>
            similarSnippets.min(prioritySnippetOrdering).snippet
        }
    }

    val resultInfo = ResultInfo.newBuilder()
      .setIsFromLongCache(false)
      .build()

    val filteredSnippets = filterHotelsByBoundingBox(foundSnippets) // TODO из-за этой строки флапают тесты

    HotelSearchResult.newBuilder()
      .setCreated(created)
      .setUpdated(System.currentTimeMillis())
      .setProgress(progress.currentProgress)
      .addAllHotelSnippet(filteredSnippets.asJava)
      .setResultInfo(resultInfo)
      .build()
  }

  private def filterHotelsByBoundingBox(snippets: Iterable[HotelSnippet]) : Iterable[HotelSnippet] = {
    val region = tree.region(request.to)
    if (region.isEmpty) {
      if (progress.isFinished && !loggedFiltered) {
        loggedFiltered = true
        log.info(s"$selfName No region found by id ${request.to}")
      }
      return snippets
    }
    val allHotelIds = snippets.map(_.getHotelId).toSet
    val allHotels = hotelsIndex.getHotelsById(allHotelIds).values
    val subRegions = tree.allChildren(region.get) + region.get
    val insideHotelIds = mutable.HashSet.empty[Int]
    for (subReg <- subRegions) yield {
      if (subReg.boundingBox.nonEmpty) {
        val insideHotels = HotelsIndex.inRectangle(subReg.boundingBox.extend(0.1), allHotels)
        for (h <- insideHotels) {
          insideHotelIds.add(h.id)
        }
      }
    }
    if (insideHotelIds.nonEmpty) {
      val (inside, outside) = snippets.partition(hs => insideHotelIds.contains(hs.getHotelId))
      if (outside.nonEmpty && progress.isFinished && !loggedFiltered) {
        loggedFiltered = true
        val filteredHotels = outside.map(s ⇒ Partners.getOpt(s.getSource(0).getPartnerId) → s.getHotelId).toMultiMap
        log.info(s"$selfName filtered ${outside.size} snippets outside region ${request.to}. Hotels: $filteredHotels")
      }
      inside
    } else {
      if (progress.isFinished && !loggedFiltered) {
        loggedFiltered = true
        log.info(s"$selfName No hotels inside region ${request.to} - ${region.map(_.name.ruName)}")
      }
      snippets
    }
  }

  override protected def getRequestToPartner(partner: Partner, ctx: Context): Option[AnyRef] = {
    val sources = allSources.filter(_.contains(partner))
    if (sources.nonEmpty) {
      Some(PartnerProtocol.SearchHotels(request, sources, ctx))
    } else {
      Option.empty
    }
  }


  /* -------------------------------------------- */

  override def receive: Receive = receivePrimary orElse super[SnippetRequestHolder].receive

  private val primaryProgress = new SearcherProgress[SnippetsResult](allSources)
  private var secondaryProgress: SearcherProgress[SnippetsResult] = _
  private var secondaryMetrics: RequestMetrics = _
  private def processMetrics(metrics: RequestMetrics, searchResult: SnippetsResult): Unit = searchResult.result match {
    case Failed(_) ⇒ metrics.failedOperator(searchResult.searchSource)
    case Successful(_) ⇒ metrics.successOperator(searchResult.searchSource)
    case Skipped ⇒ metrics.skippedOperator(searchResult.searchSource)
    case Partial(_) ⇒
  }

  private def processAvailability(searchResult: SnippetsResult): Unit = searchResult.result match {
    case Failed(_) ⇒ searchSourceAvailability.markFailure(searchResult.searchSource)
    case Successful(_) ⇒ searchSourceAvailability.markSuccess(searchResult.searchSource)
    case _ =>
  }

  private def addProgress(searchResult: SnippetsResult, partialProgress: SearcherProgress[SnippetsResult]): Unit = {
    partialProgress.add(searchResult)
    progress.add(searchResult)
  }

  private def saveResult(result: HotelSearchResult): Unit = {
    saveResponse(result) onFailure {
      case e => log.error(s"$selfName: Can not save response for $request", e)
    }
  }

  private def postProcessResultPrimary(result: HotelSearchResult): Unit = {
    if (primaryProgress.isFinished) {
      context.become(receiveSecondary orElse super[SnippetRequestHolder].receive)

      val snippets = primaryProgress.getResults.flatMap {
        _.result match {
          case r: Successful[HotelSnippet] ⇒ r.results
          case r: Partial[HotelSnippet] ⇒ r.results
          case _ ⇒ Iterable.empty[HotelSnippet]
        }
      }.toList.sortBy(s ⇒ s.getPriceMin)

      if (snippets.nonEmpty) {
        val hotelIdsToPartner = {
          val ordered = snippets.map { s ⇒
            s.getHotelId → s.getSourceList.asScala.map(_.getPartnerId)
          }
          val fullPartnersMap = ordered.toMultiMap.mapValues(_.flatten)
          ordered.map(_._1).distinct.map { hotelId ⇒ hotelId → fullPartnersMap(hotelId)}
        }

        secondaryProgress = new SearcherProgress(allSources)
        secondaryMetrics = new RequestMetrics(metricsName + "_secondary", allSources)

        storage.getContext(request.hotelRequest) onComplete {
          case Success(ctx) =>
            for {
              (partner, ref) ← partners
              r ← getRequestToPartner(partner, ctx)
            } r match {
              case SearchHotels(req, sources, c) =>
                val hotelIds = hotelIdsToPartner.filter(!_._2.contains(partner.id)).map(_._1)

                val bBox = tree.region(request.to).map(_.boundingBox).filter(_.nonEmpty)
                // filter hotels which has mappings for this partner
                val availableHotelIds = hotelsIndex
                  .getHotels(hotelIds, Seq(new PartnerIdFilter(partner.id)), bBox)
                  .values.toList.sortBy(-_.relevance).map(_.id)

                val topByPrice = hotelIds.filter(availableHotelIds.contains).take(MAX_HOTELS_TO_SEARCH)
                val topByRelevance = availableHotelIds.take(MAX_HOTELS_TO_SEARCH)
                val top = (topByPrice ::: topByRelevance).distinct

                ref ! SearchHotelsByIds(req, top, sources, c)
            }
          case Failure(e) =>
            log.error(s"$selfName Can not get context for request $request", e)
            postProcessResultSecondary(result, forceFinish = true)
        }
      } else {
        postProcessResultSecondary(result, forceFinish = true)
      }
    } else {
      saveResult(result)
    }
  }

  private def postProcessResultSecondary(result: HotelSearchResult, forceFinish: Boolean): Unit = {
    saveResult(result)
    if (forceFinish || secondaryProgress.isFinished) {
      context.system.eventStream.publish(createSearchEvent(request, result))
      context.stop(self)
    }
  }

  private def receivePrimary: Receive =
    receive(primaryProgress, metrics, postProcessResultPrimary)

  private def receiveSecondary: Receive =
    receive(secondaryProgress, secondaryMetrics, postProcessResultSecondary(_, forceFinish = false))

  private def receive(progressPartial: => SearcherProgress[PartnerProtocol.SnippetsResult],
                      metrics: => RequestMetrics,
                      postProcess: HotelSearchResult => Unit): Receive = {
    case searchResult: SnippetsResult ⇒
      /* searchResults to metrics */
      processMetrics(metrics, searchResult)
      processAvailability(searchResult)

      /* search result to progress */
      addProgress(searchResult, progressPartial)

      /* search result to result */
      val result = createResult(progress.getResults)
      postProcess(result)
    case SourcesToWait(map) ⇒
      progressPartial.updateWaitMap(map)
      progress.updateWaitMap(map)
  }

  override protected def onStop(): Unit = {
    if (secondaryProgress ne null) {
      val nonEmptyResult = secondaryProgress.getResults.exists(_.result.nonEmpty)
      metrics.finishedRequest(nonEmptyResult)
    }
    super.onStop()
  }
}

object HotelRequestHolder {
  def props[T <: Source](storage: ToursDao,
                         request: HotelSearchRequest,
                         hotelsIndex: HotelsIndex,
                         tree: Tree,
                         partners: Map[Partner, ActorRef],
                         searchSources: SearchSources[T],
                         searchSourceAvailability: SearchSourceAvailability[T]): Props = {
    Props(new HotelRequestHolder[T](storage, request, hotelsIndex, tree,
      partners, searchSources, searchSourceAvailability))
  }

  val MAX_HOTELS_TO_SEARCH = 20
}