package ru.yandex.tours.backend.search

import akka.actor.ActorRef
import ru.yandex.tours.backend.search.SearcherActor.{SpawnHotelHolder, SpawnTourHolder}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.SearchResults._
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest, SearchType}
import ru.yandex.tours.search.SearchUtil._
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.services.HotelSearchService
import ru.yandex.tours.storage.ToursDao
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.collections.SimpleBitSet
import ru.yandex.tours.util.spray.completeJsonError
import spray.http.StatusCodes

import scala.concurrent.{ExecutionContext, Future}
import scala.collection.JavaConverters._

class LocalHotelSearchService(storage: ToursDao,
                              toursCache: ToursDao,
                              searcherActor: ActorRef,
                              geoMappingHolder: GeoMappingHolder,
                              searchType: SearchType,
                              tree: Tree,
                              hotelIndex: HotelsIndex,
                              searchSettings: SearchSettingsHolder,
                              searcher2adapter: Searcher2Adapter = null
                             )
                             (implicit ec: ExecutionContext) extends HotelSearchService with Logging {

  override def search(request: HotelSearchRequest, canStartRequest: Boolean,
                      onlyFromLongCache: Boolean): Future[HotelSearchResult] = {
    val isKnown = geoMappingHolder.isKnownDestination(request.to, searchType) ||
      searchSettings.getRegionSearchSettings(request.to).forceSearchIfUnknown
    if (!isKnown) {
      Future.successful(unknownDestinationResponse)
    } else if (searchType == SearchType.TOURS && searchSettings.isTooClose(request)) {
      Future.successful(tooCloseResponse)
    } else if (onlyFromLongCache) {
      getHotelsFromCache(request, emptyHotels(emptyNotFinishedProgress), canStartRequest = false)
    } else {
      storage.getHotelSearchResult(request).flatMap[HotelSearchResult] {
        case None if canStartRequest => startHotelRequest(request)
        case None => Future.successful(emptyHotels(emptyFinishedProgress, notStarted))
        case Some(result) if isHung(result) =>
          log.info(s"Request $request is hung. Starting new one.")
          startHotelRequest(request)
        case Some(result) if isFailed(result) =>
          log.info(s"Request $request failed. Getting from cache.")
          getHotelsFromCache(request, result, canStartRequest)
        case Some(result) => Future.successful(result)
      }
    }
  }

  override def searchHotel(origRequest: OfferSearchRequest, canStartRequest: Boolean): Future[OfferSearchResult] = {
    val request = fixRequestTo(origRequest)
    storage.getOffersSearchResult(request).flatMap[OfferSearchResult] {
      case None if canStartRequest => startTourRequest(request)
      case None => Future.successful(emptyTours(emptyFinishedProgress, request.hotelId, notStarted))
      case Some(result) if isHung(result) =>
        log.info(s"Request $request is hung. Starting new one.")
        startTourRequest(request)
      case Some(result) if isFailed(result) =>
        log.info(s"Request $request failed. Getting from cache.")
        getToursFromCache(request, result, canStartRequest)
      case Some(result) => Future.successful(result)
    }
  }

  private def fixRequestTo(request: OfferSearchRequest): OfferSearchRequest = {
    if (request.hotelRequest.to != 0) {
      return request
    }
    hotelIndex.getHotelById(request.hotelId) match {
      case Some(hotel) =>
        OfferSearchRequest(request.hotelRequest.copy(to = hotel.geoId), request.hotelId)
      case None =>
        log.warn(s"Request $request has unknown hotel id ${request.hotelId}")
        request
    }
  }

  override def searchHotels(request: HotelSearchRequest, hotelIds: Iterable[Int],
                            canStartRequest: Boolean): Future[Seq[OfferSearchResult]] = {
    if (canStartRequest) {
      Future.traverse(hotelIds.toSeq) { hotelId =>
        searchHotel(OfferSearchRequest(request, hotelId), canStartRequest)
      }
    } else {
      val reqs = hotelIds.map(hotelId => fixRequestTo(OfferSearchRequest(request, hotelId)))
      val reqsByTo = reqs.groupBy(_.hotelRequest.to).toSeq
      Future.traverse(reqsByTo) { pair =>
        searchInStorage(pair._2)
      }.map(_.flatten)
    }
  }

  private def searchInStorage(reqs: Iterable[OfferSearchRequest]) : Future[Seq[OfferSearchResult]] = {
    val request = reqs.head.hotelRequest
    val hotelIds = reqs.map(_.hotelId)
    storage.getOffersSearchResults(request, hotelIds).map { results =>
      val map = results.map(r => r.getHotelId -> r).toMap
      for (hotelId <- hotelIds.toSeq)
        yield map.getOrElse(hotelId, emptyTours(emptyFinishedProgress, hotelId, notStarted))
    }
  }

  private def startTourRequest(request: OfferSearchRequest): Future[OfferSearchResult] = {
    if (searcher2adapter!=null) searcher2adapter.search(request)
    val result = emptyTours(emptyNotFinishedProgress, request.hotelId)
    storage.saveOffersSearchResult(request, result).map(_ => {
      searcherActor ! SpawnTourHolder(request)
      result
    })
  }

  private def getToursFromCache(request: OfferSearchRequest, newResult: OfferSearchResult,
                                canStartRequest: Boolean = false): Future[OfferSearchResult] = {
    toursCache.getOffersSearchResult(request).flatMap {
      case None if canStartRequest => startTourRequest(request)
      case Some(result) if isFailed(result) && canStartRequest => startTourRequest(request)
      case Some(result) if isFailed(newResult) =>
        val failedOperators = SimpleBitSet.from(newResult.getProgress.getOperatorFailedSet).toSet
        val offers = result
          .getOfferList
          .asScala
          .filter(offer => failedOperators.contains(offer.getSource.getOperatorId))
        val fixedResult = result.toBuilder.clearOffer().addAllOffer(offers.asJava).build()
        Future.successful(fixedResult)
      case Some(result) => Future.successful(result)
      case _ => Future.successful(newResult)
    }
  }

  private def getHotelsFromCache(request: HotelSearchRequest, newResult: HotelSearchResult,
                                 canStartRequest: Boolean = false): Future[HotelSearchResult] = {
    toursCache.getHotelSearchResult(request).flatMap {
      case None if canStartRequest => startHotelRequest(request)
      case Some(result) if isFailed(result) && canStartRequest => startHotelRequest(request)
      case Some(result) if isFailed(newResult) =>
        val failedOperators = SimpleBitSet.from(newResult.getProgress.getOperatorFailedSet).toSet
        val snippets = result
          .getHotelSnippetList
          .asScala
          .filter { snippet =>
            failedOperators.contains(snippet.getSource(0).getOperatorId)
          }
        val fixedResult = result.toBuilder.clearHotelSnippet().addAllHotelSnippet(snippets.asJava).build()
        Future.successful(fixedResult)
      case Some(result) => Future.successful(result)
      case _ => Future.successful(newResult)
    }
  }

  private def startHotelRequest(request: HotelSearchRequest): Future[HotelSearchResult] = {
    val result = emptyHotels(emptyNotFinishedProgress)
    // Wait for saving result and then reply.
    storage.saveHotelSearchResult(request, result).map(_ => {
      searcherActor ! SpawnHotelHolder(request)
      result
    })
  }

  private val resultInfo = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .build()

  private val unknownDestination = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.UNKNOWN_DESTINATION)
    .build()

  private val notStarted = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.NOT_STARTED)
    .build()

  private val tooClose = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.TOO_CLOSE_DESTINATION)
    .build()

  protected val emptyProgressBuilder = SearchProgress.newBuilder()
    .setIsFinished(false)
    .setOperatorCompleteCount(0)
    .setOperatorTotalCount(0)
    .setOperatorFailedCount(0)
    .setOperatorSkippedCount(0)

  private val emptyFinishedProgress = emptyProgressBuilder.setIsFinished(true).build()

  private val emptyNotFinishedProgress = emptyProgressBuilder.setIsFinished(false).build()

  private def emptyHotels(progress: SearchProgress, resultInfo: ResultInfo = resultInfo) =
    HotelSearchResult.newBuilder()
      .setProgress(progress)
      .setCreated(System.currentTimeMillis())
      .setUpdated(System.currentTimeMillis())
      .setResultInfo(resultInfo)
      .build()

  private def emptyTours(progress: SearchProgress, hotelId: Int, resultInfo: ResultInfo = resultInfo) =
    OfferSearchResult.newBuilder()
      .setProgress(progress)
      .setCreated(System.currentTimeMillis())
      .setUpdated(System.currentTimeMillis())
      .setHotelId(hotelId)
      .setResultInfo(resultInfo)
      .build()

  private def unknownDestinationResponse =
    HotelSearchResult.newBuilder()
      .setProgress(emptyFinishedProgress)
      .setCreated(System.currentTimeMillis())
      .setUpdated(System.currentTimeMillis())
      .setResultInfo(unknownDestination)
      .build

  private def tooCloseResponse =
    HotelSearchResult.newBuilder()
      .setProgress(emptyFinishedProgress)
      .setCreated(System.currentTimeMillis())
      .setUpdated(System.currentTimeMillis())
      .setResultInfo(tooClose)
      .build
}
