package ru.yandex.tours.backend.search

import akka.actor.ActorRef
import ru.yandex.tours.backend.search.SearcherActor.SpawnActualizeHolder
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.SearchResults._
import ru.yandex.tours.model.search._
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.services.OfferSearchService
import ru.yandex.tours.storage.ToursDao

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future, Promise}
import scala.util.Success

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 26.01.15
 */
class LocalOffersSearchService(storage: ToursDao,
                               toursCache: ToursDao,
                               searcherActor: ActorRef,
                               geoMappingHolder: GeoMappingHolder,
                               tree: Tree,
                               hotelIndex: HotelsIndex,
                               searchSettings: SearchSettingsHolder,
                               searcher2adapter: Searcher2Adapter = null)(implicit ec: ExecutionContext)
  extends LocalHotelSearchService(storage, toursCache, searcherActor, geoMappingHolder, SearchType.TOURS, tree, hotelIndex, searchSettings, searcher2adapter)
    with OfferSearchService {

  override def searchOffer(request: GetOfferRequest): Future[Option[ActualizedOffer]] = {
    storage.getOffersSearchResult(request.toursInHotelRequest).map {
      case Some(result) =>
        result.getOfferList.asScala.find(request.matches).map { offer =>
          ActualizedOffer.newBuilder()
            .setCreated(System.currentTimeMillis())
            .setOffer(offer)
            .setResultInfo(result.getResultInfo)
            .build()
        }
      case None => None
    }
  }

  override def actualizeOffer(request: GetOfferRequest): Future[Option[ActualizedOffer]] = {
    val promise = Promise.apply[Option[ActualizedOffer]]()
    searchOffer(request).onComplete {
      case Success(Some(tourOffer)) =>
        searcherActor ! SpawnActualizeHolder(request, tourOffer, promise)
      case _ =>
        promise.success(None)
    }
    promise.future
  }
}
