package ru.yandex.tours.backend.search

import akka.actor.{ActorRef, Props}
import org.joda.time.DateTime
import ru.yandex.tours.events.SearchEvent.FoundOffers
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.OfferSearchRequest
import ru.yandex.tours.model.search.SearchProducts.Offer
import ru.yandex.tours.model.search.SearchResults.{Context, OfferSearchResult, ResultInfo}
import ru.yandex.tours.operators.{SearchSourceAvailability, SearchSources}
import ru.yandex.tours.partners.PartnerProtocol
import ru.yandex.tours.partners.PartnerProtocol.{OffersResult, Partial, Successful}
import ru.yandex.tours.storage.ToursDao

import scala.collection.JavaConverters._
import scala.concurrent.Future

class OfferRequestHolder[S <: Source](storage: ToursDao,
                                      request: OfferSearchRequest,
                                      partners: Map[Partner, ActorRef],
                                      searchSources: SearchSources[S],
                                      searchSourceAvailability: SearchSourceAvailability[S])
  extends SnippetRequestHolder[OfferSearchRequest, OfferSearchResult, S, OffersResult](storage, request, partners,
    searchSources, searchSourceAvailability) {

  override protected def metricsName: String = "tours_search"

  override protected def getRequestToPartner(partner: Partner, ctx: Context): Option[AnyRef] = {
    val operators = allSources.filter(_.contains(partner))
    if (operators.nonEmpty) {
      Some(PartnerProtocol.SearchOffers(request, operators, ctx))
    } else {
      Option.empty
    }
  }

  override protected def saveResponse(res: OfferSearchResult): Future[Unit] =
    storage.saveOffersSearchResult(request, res)

  override protected def createSearchEvent(request: OfferSearchRequest, res: OfferSearchResult) =
    FoundOffers(DateTime.now, request, res)

  private case class KeyedOffer(offer: Offer, priority: Int) {
    val key = {
      (
        offer.getSource.getOperatorId,
        offer.getDate, offer.getNights,
        offer.getRoomType, offer.getPansion,
        offer.getWithTransfer,
        offer.getWithFlight,
        offer.hasFreeCancellation,
        offer.getFreeCancellation
      )
    }
  }

  private implicit val keyedOfferOrdering = Ordering.by[KeyedOffer, (Int, Int)] {
    keyedOffer ⇒ (keyedOffer.priority, keyedOffer.offer.getPrice)
  }

  override protected def createResult(results: Iterable[OffersResult]): OfferSearchResult = {
    val rawOffers = results.flatMap {
      case OffersResult(_, Successful(offers), priority) ⇒ offers.map(offer ⇒ KeyedOffer(offer, priority))
      case OffersResult(_, Partial(offers), priority) ⇒ offers.map(offer ⇒ KeyedOffer(offer, priority))
      case _ ⇒ Iterable.empty
    }
    val distinctOffers = rawOffers.groupBy(_.key)
      .map { case (_, sameOffers) ⇒ sameOffers.min.offer }
    val resultInfo = ResultInfo.newBuilder()
      .setIsFromLongCache(false)
      .build()

    OfferSearchResult.newBuilder()
      .setCreated(created)
      .setUpdated(System.currentTimeMillis())
      .setProgress(progress.currentProgress)
      .setHotelId(request.hotelId)
      .setResultInfo(resultInfo)
      .addAllOffer(distinctOffers.asJava)
      .build()
  }
}

object OfferRequestHolder {
  def props[S <: Source](storage: ToursDao,
                         request: OfferSearchRequest,
                         partners: Map[Partner, ActorRef],
                         searchSources: SearchSources[S],
                         searchSourceAvailability: SearchSourceAvailability[S]): Props = {
    Props(new OfferRequestHolder(storage, request, partners, searchSources, searchSourceAvailability))
  }
}