package ru.yandex.tours.backend.search

import java.util.UUID

import io.grpc.{ManagedChannel, ManagedChannelBuilder}
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.Futures._
import ru.yandex.travel.proto._

import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success}

class Searcher2Adapter(host: String, port: Int, hotelsIndex: HotelsIndex)
                      (implicit ec: ExecutionContext) extends Logging {
  private val channel: ManagedChannel = ManagedChannelBuilder.forAddress(host, port).usePlaintext().build()
  private val futureStub: OfferSearchServiceV1Grpc.OfferSearchServiceV1FutureStub =
    OfferSearchServiceV1Grpc.newFutureStub(channel)

  private val partnerMap: Map[Partners.Value, EPartnerId] = Map(
    Partners.ostrovokv3 -> EPartnerId.PI_OSTROVOK,
    Partners.booking -> EPartnerId.PI_BOOKING,
    Partners.hotels101 -> EPartnerId.PI_HOTELS101,
    Partners.hotelsCombined -> EPartnerId.PI_HOTELSCOMBINED,
    Partners.travelline -> EPartnerId.PI_TRAVELLINE
  )

  private def getOccupancy(req: HotelSearchRequest) = {
    val bldr = StringBuilder.newBuilder.append(req.adults)
    if (req.getAges.kids > 0) {
      bldr.append("-")
      bldr.append(req.kidsAges.mkString(","))
    }
    bldr.toString()
  }

  def search(request: OfferSearchRequest): Future[Unit] = {
    val builder = TSearchOffersRpcReq.newBuilder()
    builder.setSync(false)
    builder.setIncludeDebug(true)
    for (hotel <- hotelsIndex.getHotelById(request.hotelId)) {
      for (partnerInfo <- hotel.partnerIds) {
        for (partner <- partnerMap.get(partnerInfo.partner)) {
          val offer = TSearchOffersReq.newBuilder()
            .setPartnerId(partner)
            .setOriginalId(partnerInfo.id)
            .setCheckInDate(request.hotelRequest.when.toString("YYYY-MM-dd"))
            .setCheckOutDate(request.hotelRequest.whenBack.toString("YYYY-MM-dd"))
            .setOccupancy(getOccupancy(request.hotelRequest))
            .setCurrency(ECurrency.C_RUB)
            .setRequestClass(ERequestClass.RC_BACKGROUND)
            .setId(request.hotelRequest.uniqueId)
            .setUtmSource(request.hotelRequest.utmSource.getOrElse("old_searcher"))
            .setUtmMedium(request.hotelRequest.utmMedium.getOrElse(""))
            .setUtmCampaign(request.hotelRequest.utmCampaign.getOrElse(""))
            .setUtmTerm(request.hotelRequest.utmTerm.getOrElse(""))
            .setUtmContent(request.hotelRequest.utmContent.getOrElse(""))
            .build()
          builder.addSubrequest(offer)
        }
      }
    }
    if (builder.getSubrequestCount > 0) {
      val f = futureStub.searchOffers(builder.build()).asScalaFutureUnit
      f.onComplete {
        {
          case Success(res) =>
            log.info("Successfully sent request to Searcher2")
          case Failure(res) =>
            log.error("Could not sent a request to Searcher2", res)
        }
      }
      f
    }
    else {
      Future.successful(())
    }
  }
}
