package ru.yandex.tours.backend.search

import akka.actor._
import ru.yandex.tours.backend.search.SearcherActor.{SpawnActualizeHolder, SpawnHotelHolder, SpawnTourHolder}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchResults._
import ru.yandex.tours.model.search.{GetOfferRequest, HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.operators.{SearchSourceAvailability, SearchSources}
import ru.yandex.tours.storage.ToursDao
import ru.yandex.tours.util.{Logging, Metrics}

import scala.concurrent.Promise
import scala.util.control.NonFatal

/* @author berkut@yandex-team.ru */

object SearcherActor {

  case class SpawnHotelHolder(request: HotelSearchRequest)

  case class SpawnTourHolder(request: OfferSearchRequest)

  case class SpawnActualizeHolder(request: GetOfferRequest, tourOffer: ActualizedOffer, promise: Promise[Option[ActualizedOffer]])

  case class SearchHotelSnippets(request: HotelSearchRequest, onlyFromCache: Boolean)

}

class SearcherActor[T <: Source](storage: ToursDao,
                                 partners: Map[Partner, ActorRef],
                                 searchSources: SearchSources[T],
                                 hotelsIndex: HotelsIndex,
                                 tree: Tree,
                                 searchSourceAvailability: SearchSourceAvailability[T]) extends Actor with Logging {

  private var actualizeId = 0

  Metrics("actors." + self.path.name).newGauge("children") { context.children.size }

  override def receive: Receive = {
    case SpawnHotelHolder(request) =>
      val childName = request.sessionId
      context.child(childName).getOrElse {
        val props = HotelRequestHolder.props(storage, request, hotelsIndex, tree, partners,
          searchSources, searchSourceAvailability)
        context.actorOf(props, childName)
      }

    case SpawnTourHolder(request) =>
      val childName = request.requestId
      context.child(childName).getOrElse {
        val props = OfferRequestHolder.props(storage, request, partners,
          searchSources, searchSourceAvailability)
        context.actorOf(props, childName)
      }

    case SpawnActualizeHolder(request, tourOffer, promise) =>
      actualizeId += 1
      val childName = s"${request.hotelRequest.sessionId}-${request.tourId}-$actualizeId"
      context.actorOf(Props(new ActualizeRequestHolder(storage, request, tourOffer, promise, partners)), childName)
  }

  override def supervisorStrategy: SupervisorStrategy = OneForOneStrategy() {
    case NonFatal(_) ⇒ SupervisorStrategy.Stop
  }
}
