package ru.yandex.tours.backend.search

import akka.actor.{ActorRef, ReceiveTimeout}
import ru.yandex.tours.events.SearchEvent
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.BaseRequest
import ru.yandex.tours.operators.{SearchSourceAvailability, SearchSources}
import ru.yandex.tours.partners.PartnerProtocol._
import ru.yandex.tours.storage.ToursDao
import ru.yandex.tours.util.Collections._

import scala.concurrent.Future
import scala.concurrent.duration._
import scala.reflect.ClassTag

abstract class SnippetRequestHolder[T <: BaseRequest,
                                    R <: AnyRef,
                                    S <: Source,
                                    I <: SearchResult[_]: ClassTag]
                                    (storage: ToursDao,
                                     request: T,
                                     partners: Map[Partner, ActorRef],
                                     searchSources: SearchSources[S],
                                     searchSourceAvailability: SearchSourceAvailability[S])
  extends AbstractRequestHolder[T, R](storage, request, partners) {

  import context.dispatcher

  protected def metricsName: String
  protected val allSources = searchSources.getForPartners(partners.keySet).toSet
  protected val progress = new SearcherProgress[I](allSources)
  protected val created = System.currentTimeMillis()
  protected val metrics = new RequestMetrics(metricsName, allSources)

  private val hangTimeout = 2.minutes

  protected def saveResponse(res: R): Future[Unit]

  protected def createResult(results: Iterable[I]): R

  protected def createSearchEvent(request: T, res: R): SearchEvent

  override def receive: Receive = {
    case UpdateContext(ctx) =>
      storage.saveContext(request.hotelRequest, ctx) onFailure {
        case e => log.error(s"Can not update context for $request", e)
      }
    case searchResult: I =>
      searchResult.result match {
        case Failed(_) =>
          metrics.failedOperator(searchResult.searchSource)
          searchSourceAvailability.markFailure(searchResult.searchSource)
        case Successful(_) =>
          metrics.successOperator(searchResult.searchSource)
          searchSourceAvailability.markSuccess(searchResult.searchSource)
        case Skipped => metrics.skippedOperator(searchResult.searchSource)
        case _ =>
      }
      progress.add(searchResult)
      val result = createResult(progress.getResults)
      saveResponse(result) onFailure {
        case e => log.error(s"Can not save response for $request", e)
      }
      if (progress.isFinished) {
        context.system.eventStream.publish(createSearchEvent(request, result))
        context.stop(self)
      }
    case SourcesToWait(map) =>
      progress.updateWaitMap(map)
    case ReceiveTimeout =>
      log.warn(s"$selfName seems hang. Stopping...")
      log.warn(s"$selfName current progress is $progress")
      context.stop(self)
  }


  override def preStart(): Unit = {
    super.preStart()
    context.setReceiveTimeout(hangTimeout)
  }

  override protected def onStop(): Unit = {
    val nonEmptyResult = progress.getResults.exists(!_.result.isEmpty)
    metrics.finishedRequest(nonEmptyResult)
    dumpErrors()
  }

  private def dumpErrors(): Unit = {
    val errors = progress.getResults.flatMap { result =>
      result.result match {
        case Failed(e) => Some(result.searchSource -> e)
        case _ => None
      }
    }.toMap
    if (errors.nonEmpty) {
      val sources = errors.keys.map(_.name)
      log.info(s"$selfName failed to fetch data from ${errors.size} sources for $request: ${sources.mkString(",")}")
      if (errors.keySet.size > allSources.size / 2 || log.isDebugEnabled) {
        val grouped = errors.toSeq.swap.toMultiMap
        for ((error, operators) <- grouped) {
          log.warn(s"Sources ${operators.mkString(", ")} failure", error)
        }
      }
    }
  }
}
