package ru.yandex.tours.backend.transfer

import akka.util.Timeout
import com.typesafe.config.Config
import ru.yandex.tours.avia.Airports
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.TransferSearchRequest
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang._

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Try}
import spray.http.{StatusCode, StatusCodes, Uri}

/**
  * Created by asoboll on 25.04.16.
  */
class CarTrawlerClient(hotelIndex: HotelsIndex,
                       airports: Airports,
                       httpClient: AsyncHttpClient,
                       config: Config,
                       timeout: Timeout = Timeout(60.seconds))
                      (implicit val ec: ExecutionContext) extends TransferClient with Logging {
  val clientName = "cartrawler"

  implicit private val implicitTimeout = timeout

  private val parser = new CarTrawlerParser

  private val baseUri = config.getString("base_uri")
  private val clientId = config.getString("client_id")
  private val lang = Try(config.getString("lang")).getOrElse("RU")
  private val currency = Try(config.getString("currency")).getOrElse("RUB")

  override def searchTransfers(transferSearchRequest: TransferSearchRequest): Future[TransferClient.Response] = {
    for {
      uri <- getUri(transferSearchRequest).toFuture
      (sc, response) <- httpClient.get(uri)
      transfers <- processResponse(transferSearchRequest, sc, response).toFuture
    } yield transfers.copy(requestUrl = Option(uri.toString))
  }

  private def getIataCode(id: String): String = {
    val airportOpt = airports.byId(id)
    airportOpt.flatMap(_.iata).getOrElse {
      val errorMsg = s"iata code not found for airportId: $id, airport: $airportOpt"
      log.warn(errorMsg)
      throw new NoSuchElementException(errorMsg)
    }
  }

  private def getUri(request: TransferSearchRequest): Try[Uri] = Try {
    val airport = getIataCode(request.airportId)
    val hotel = hotelIndex.getHotelById(request.hotelId).get
    var requestParams = Map(
      "clientId" -> clientId,
      "residencyId" -> "RU",
      "pkDateTime" -> CarTrawlerParser.whenStr(request),
      "rtDateTime" -> CarTrawlerParser.whenBackStr(request),
      "locCode" -> airport,
      "destRefPoint" -> s"${hotel.latitude},${hotel.longitude}",
      "passNum" -> request.hotelRequest.ages.size.toString,
      "lang" -> lang,
      "currency" -> currency
    )
    request.userIp.foreach(requestParams += "ip" -> _)

    Uri(baseUri).withQuery(requestParams)
  } onSuccess { uri =>
    log.debug("transfer url: " + uri.toString)
  }

  private def processResponse(request: TransferSearchRequest,
                             statusCode: StatusCode,
                             response: String): Try[TransferClient.Response] = {
    log.debug(s"answer from CarTrawler: ${statusCode.toString}, $response \nfor request: $request")
    statusCode match {
      case StatusCodes.OK => parser.parse(request, response)
      case _ => Failure(new Exception(s"Status code is not 200 for request [$request] from CarTrawler: $statusCode"))
    }
  }
}
