package ru.yandex.tours.backend.transfer

import java.io.ByteArrayInputStream

import org.joda.time.format.DateTimeFormat
import ru.yandex.common.xml.{XParser, XTag}
import ru.yandex.tours.model.search.SearchProducts.TransferOption
import ru.yandex.tours.model.search.TransferSearchRequest
import ru.yandex.tours.util.parsing.IntValue

import scala.collection.mutable.ArrayBuffer
import scala.util.{Failure, Success, Try}
import scala.collection.JavaConversions._

/**
  * Created by asoboll on 25.04.16.
  */
class CarTrawlerParser {
  import CarTrawlerParser._

  private val DEFAULT_CURRENCY = "RUB"

  private def unifyType(t: String, hasPriceDaily: Boolean): String = {
    import CarTrawlerParser.TransferTypes._

    if (hasPriceDaily) RENTAL_TYPE_NAME
    else t.toLowerCase match {
      case s if PUBLIC_TYPES.exists(s.contains) => PUBLIC_TYPE_NAME
      case s if TAXI_TYPES.exists(s.contains) => TAXI_TYPE_NAME
      case s => throw new IllegalArgumentException(s"unknown transfer type: $s")
    }
  }

  def parse(request: TransferSearchRequest, response: String): Try[TransferClient.Response] = Try {
    val filteredResponse = response
      .replaceAllLiterally("&nbsp;", " ")
      .replaceAllLiterally("&mdash;", "&#x2014;")
      .replaceAllLiterally("&ndash;", "&#x2013;")

    val transfers = ArrayBuffer.empty[TransferOption]
    val errors = ArrayBuffer.empty[TransferClient.TransferParsingException]

    new XParser {
      override def build(): XTag = new XTag() {
        tag("products") = handle {
          tag("product") = handle {
            val builder = TransferOption.newBuilder

            complete {
              Try {
                builder.setPickupTime(whenStr(request))
                builder.setDropoffTime(whenBackStr(request))
                builder.setUnifiedType(unifyType(builder.getType, builder.hasPriceDaily))
                builder.build
              }.flatMap(validateTransfer(_, request)) match {
                case Success(transfer) => transfers += transfer
                case Failure(e: Exception) => errors += TransferClient.TransferParsingException(e)
                case Failure(e) => throw e
              }
            }

            tag("type") = handleComplete(builder.setType(text))
            tag("displayname") = handleComplete(builder.setName(text))
            tag("detailname") = handleComplete(if (text.nonEmpty) builder.setNameDetailed(text))

            tag("pickuplocation") = handleComplete(builder.setPickupLoc(text))
            tag("dropofflocation") = handleComplete(builder.setDropoffLoc(text))

            tag("image") = handleComplete(builder.setImage(text))
            tag("vendorimage") = handleComplete(builder.setImageVendor(text))

            tag("totalprice") = handleComplete {
              if (attr("Currency") == DEFAULT_CURRENCY)
                builder.setPrice(text.toDouble)
            }
            tag("dailyprice") = handleComplete {
              if (attr("Currency") == DEFAULT_CURRENCY)
                builder.setPriceDaily(text.toDouble)
            }

            tag("deeplink") = handleComplete {
              builder.setLink(text)
            }

            tag("details") = handle {
              tag("detail") = handleComplete {
                val name = attr("Name")
                val value = attr("Value")
                val detail = TransferOption.Detail.newBuilder.setName(name).setValue(value).build
                builder.addDetails(detail)
              }
            }
          }
        }
      }
    }.setLowerCase(true).parse(new ByteArrayInputStream(filteredResponse.getBytes))

    TransferClient.Response(transfers, errors)
  }

  def validateTransfer(transfer: TransferOption, request: TransferSearchRequest): Try[TransferOption] = Try {
    require(transfer.getType.nonEmpty)
    require(transfer.getName.nonEmpty)
    require(transfer.getLink.nonEmpty)
    require(transfer.getPrice > 0.0d)
    require(transfer.getUnifiedType.nonEmpty)
    require((for {
      passNumDetail <- transfer.getDetailsList.find(_.getName == "passengerQuantity")
      passNum <- IntValue(passNumDetail.getValue)
    } yield passNum >= request.hotelRequest.ages.size).getOrElse(true))

    transfer
  }
}

object CarTrawlerParser {
  private def dateFormat = DateTimeFormat.forPattern("yyyyMMdd")
  def whenStr(request: TransferSearchRequest): String = s"${dateFormat.print(request.hotelRequest.when)}1000"
  def whenBackStr(request: TransferSearchRequest): String = s"${dateFormat.print(request.hotelRequest.whenBack)}1000"

  private object TransferTypes {
    val RENTAL_TYPE_NAME = "rental"
    val PUBLIC_TYPE_NAME = "public"
    val TAXI_TYPE_NAME = "taxi"

    val PUBLIC_TYPES = Seq("train")
    val TAXI_TYPES = Seq("boat", "minibus", "sedan", "suv", "stretch_limo", "convertible", "shuttle", "taxi", "coach")
  }
}