package ru.yandex.tours.backend.transfer

import akka.actor.{ActorRef, ActorSystem}
import org.joda.time.DateTime
import ru.yandex.tours.events.SearchEvent.FoundTransfers
import ru.yandex.tours.model.search.SearchResults.{ErrorCode, ResultInfo, SearchProgress, TransferSearchResult}
import ru.yandex.tours.model.search.TransferSearchRequest
import ru.yandex.tours.services.TransferSearchService
import ru.yandex.tours.storage.transfer.TransferDao
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.collections.SimpleBitSet

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future, Promise}
import scala.util.{Failure, Success}

/**
  * Created by asoboll on 25.04.16.
  */
class LocalTransferSearchService(transferDao: TransferDao,
                                 rootSearcherActor: ActorRef)
                                (implicit ec: ExecutionContext, akkaSystem: ActorSystem)
  extends TransferSearchService with Logging {

  override def searchTransfers(request: TransferSearchRequest,
                               canStartRequest: Boolean): Future[TransferSearchResult] = {
    transferDao.getTransfers(request).flatMap {
      case Some(result) => Future.successful(result)
      case None if canStartRequest => startNewRequest(request)
      case None =>
        Future.successful {
          BuildTransfers(System.currentTimeMillis(), Option.empty, successProgress, notStarted)
        }
    }
  }

  private def startNewRequest(request: TransferSearchRequest): Future[TransferSearchResult] = {
    val started = System.currentTimeMillis()
    val result = BuildTransfers(started, Option.empty, emptyProgress)
    val promise = Promise.apply[TransferClient.Response]()

    promise.future.onComplete {
      case Success(transfers) =>
        val res = BuildTransfers(started, Some(transfers), successProgress)
        akkaSystem.eventStream.publish(FoundTransfers(DateTime.now, request, res))
        transferDao.saveTransfers(request, res)
      case Failure(t) =>
        log.warn(s"Failed to search transfers. Request = $request", t)
        val res = BuildTransfers(started, Option.empty, failedProgress, allFailed)
        transferDao.saveTransfers(request, res)
    }

    transferDao.saveTransfers(request, result).map { _ =>
      rootSearcherActor ! RootTransferSearcherActor.SpawnSearcher(request, promise)
      result
    }
  }

  private def BuildTransfers(started: Long, transferResponse: Option[TransferClient.Response], progress: SearchProgress,
                             resultInfo: ResultInfo = defaultResultInfo): TransferSearchResult = {
    val resultBuilder = TransferSearchResult.newBuilder()
      .setCreated(started)
      .setUpdated(System.currentTimeMillis())
      .setProgress(progress)
      .setResultInfo(resultInfo)
      .addAllTransferOption(transferResponse.toSeq.flatMap(_.transfers).asJava)
    transferResponse.flatMap(_.requestUrl).foreach(resultBuilder.setRequestUrl)
    resultBuilder.build()
  }

  private val emptyProgress = progress(0, 0, 1)
  private val successProgress = progress(1, 0, 1)
  private val failedProgress = progress(0, 1, 1)

  private def progress(success: Int, failure: Int, total: Int) = SearchProgress.newBuilder()
    .setIsFinished(success + failure >= total)
    .setOperatorTotalCount(total)
    .setOperatorCompleteCount(success + failure)
    .setOperatorFailedCount(failure)
    .setOperatorSkippedCount(0)
    .setOperatorSkippedSet(SimpleBitSet(Set.empty).packed)
    .build()

  private val defaultResultInfo = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .build()

  private val notStarted = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.NOT_STARTED)
    .build()

  private val allFailed = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.ALL_SOURCES_FAILED)
    .build()
}
