package ru.yandex.tours.backend.transfer

import akka.actor.{Actor, Props, Status}
import akka.pattern._
import ru.yandex.tours.model.search.TransferSearchRequest
import ru.yandex.tours.util.Logging

import scala.concurrent.Promise
import scala.util.Try

/**
  * Created by asoboll on 25.04.16.
  */
class TransferSearcher(transferClient: TransferClient,
                       request: TransferSearchRequest,
                       promise: Promise[TransferClient.Response]) extends Actor with Logging {

  import context.dispatcher

  transferClient.searchTransfers(request).pipeTo(self)

  override def receive: Receive = {
    case res: TransferClient.Response =>
      complete(res)
  }

  override def unhandled(message: Any): Unit = message match {
    case Status.Failure(t) => failure(t)
    case _ => super.unhandled(message)
  }

  private def complete(res: TransferClient.Response): Unit = {
    log.info(s"Found ${res.transfers.size} transfers for request $request")
    if (res.errors.nonEmpty) {
      log.warn(s"Cannot parse ${res.errors.size} transfers for request $request")
      for (e <- res.errors) {
        log.debug("Transfer parsing error", e.e)
      }
    }
    promise.complete(Try(res))
    context.stop(self)
  }

  private def failure(t: Throwable): Unit = {
    promise.failure(t)
    log.error("Transfer search failed with exception", t)
    context.stop(self)
  }
}

object TransferSearcher {
  def props(transferClient: TransferClient,
            request: TransferSearchRequest,
            promise: Promise[TransferClient.Response]): Props = {
    Props(new TransferSearcher(transferClient, request, promise))
  }
}