package ru.yandex.tours.partners

import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.Monitorings
import ru.yandex.tours.util.http.{HttpMetrics, HttpTimeout}
import spray.http.StatusCodes

import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success}

trait InstrumentedPartnerClient {
  protected def getWarningRate(partner: Partner): Double = {
    partner match {
      case Partners.sodis => 0.5
      case Partners.lt => 0.5
      case _ => 0.1
    }
  }

  protected def getMaxRate(partner: Partner): Double = {
    partner match {
      case Partners.sodis => 0.85
      case Partners.lt => 0.85
      case _ => 0.4
    }
  }

  protected def wrap[T](partner: Partner, name: String, future: => Future[T])
                       (implicit ec: ExecutionContext): Future[T] = {
    val wRate = getWarningRate(partner)
    val mRate = getMaxRate(partner)
    val metrics = new HttpMetrics(s"partners.$partner.http.$name", warningRate = wRate, maxRate = mRate, size = 1000)
    val monitoringTimeouts = Monitorings(s"partners.$partner.timeout.$name", devOnly = true)
      .errorRate("errors", wRate, mRate, size = 1000, minErrorCount = 10)
    val ctx = metrics.start()
    future.andThen {
      case Success(_) =>
        ctx.finish(StatusCodes.OK)
        monitoringTimeouts.ok()
      case Failure(HttpTimeout(e)) =>
        metrics.timeout()
        monitoringTimeouts.error(e)
      case Failure(e) =>
        ctx.finish(StatusCodes.ServiceUnavailable, Some(e))
        monitoringTimeouts.error(e)
    }
  }
}
