package ru.yandex.tours.partners

import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.{ExtendedBaseRequest, BaseRequest}
import ru.yandex.tours.util.Logging

import scala.collection.mutable
import scala.reflect.ClassTag
import scala.util.{Success, Failure, Try}

object PartnerParsingUtil extends Logging {
  def translateAndLogErrors[X, Y : ClassTag](objects: Iterable[X], partner: Partner, request: ExtendedBaseRequest)
                                            (parse: (X, ExtendedBaseRequest) => Try[ParsingResult[Y]]): Iterable[Y] = {

    val unknownHotels = mutable.Buffer.empty[String]
    val failedByPansion = mutable.Buffer.empty[String]
    val unknownPansions = mutable.Set.empty[String]
    val failures = mutable.HashMap.empty[X, Throwable]

    var failedCount = 0
    val result = objects.flatMap { obj => parse(obj, request) match {
      case Success(UnknownPansion(pansion)) =>
        failedCount += 1
        failedByPansion += pansion
        unknownPansions += pansion
        None
      case Success(UnknownHotelId(hotelId)) =>
        failedCount += 1
        unknownHotels += hotelId
        None
      case Success(Skipped) =>
        None
      case Failure(e) =>
        failedCount += 1
        failures += obj -> e
        None
      case Success(ParsedWithDefaultPansion(x, pansion)) =>
        unknownPansions += pansion
        Some(x)
      case Success(Parsed(x)) =>
        Some(x)
      case x =>
        log.warn(s"Unexpected parsing result: $x for request $request")
        None
    }}
    if (failedCount > 0) {
      val error =
        new StringBuilder(s"Can not parse some $partner objects for request: $request. Total errors: $failedCount")

      if (failedByPansion.nonEmpty) {
        error ++= s"\nUnknown pansion: ["
        failedByPansion.distinct.addString(error, ", ")
        error ++= "] for "
        error append failedByPansion.size append " snippets"
      }
      if (unknownHotels.nonEmpty) {
        error ++= s"\nUnknown ${unknownHotels.toSet.size} hotels"
        if (log.isDebugEnabled) {
          error ++= s": ${unknownHotels.distinct.mkString(", ")}"
        }
      }
      if (failures.nonEmpty) {
        error ++= s"\n${failures.size} parsing failures."
        if (log.isDebugEnabled) {
          for ((x, message) ← failures) {
            error ++= s"\n${message.getClass.getName} ${message.getMessage} from `$x`"
          }
        }
      }
      log.warn(error.toString())
    }
    if (unknownPansions.nonEmpty) {
      log.warn(s"Unknown pansions for request $request to $partner: [${unknownPansions.mkString(", ")}]")
    }

    result
  }

  sealed trait ParsingResult[+T]

  case class UnknownPansion(pansion: String) extends ParsingResult[Nothing]

  case class UnknownHotelId(id: String) extends ParsingResult[Nothing]

  case class ParsedWithDefaultPansion[X](obj: X, pansion: String) extends ParsingResult[X]

  case class Parsed[X](obj: X) extends ParsingResult[X]

  case object Skipped extends ParsingResult[Nothing]

}
