package ru.yandex.tours.partners

import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.search.SearchResults.{Context, ActualizedOffer}
import ru.yandex.tours.model.search.{HotelSearchRequest, GetOfferRequest, OfferSearchRequest}

import scala.util.Try

object PartnerProtocol {
  /*
   * Requests
   */
  case class SearchHotels[S <: Source](request: HotelSearchRequest, operators: Set[S], context: Context)

  case class SearchHotelsByIds[S <: Source](request: HotelSearchRequest, hotelIds: List[Int], operators: Set[S], context: Context)

  case class SearchOffers[S <: Source](request: OfferSearchRequest, operators: Set[S], context: Context)

  case class ActualizeTourOffer(request: GetOfferRequest, tourOffer: ActualizedOffer)

  /*
   * Responses
   */
  sealed trait Results[+T] {
    def isEmpty: Boolean
    final def nonEmpty: Boolean = !isEmpty
  }

  case class Successful[T](results: Iterable[T]) extends Results[T] {
    override def isEmpty: Boolean = results.isEmpty
  }

  case object Skipped extends Results[Nothing] {
    override def isEmpty: Boolean = true
  }

  case class Failed(exception: Throwable) extends Results[Nothing] {
    override def isEmpty: Boolean = true
  }

  case class Partial[T](results: Iterable[T]) extends Results[T] {
    override def isEmpty: Boolean = results.isEmpty
  }

  sealed trait Response

  /*
   * Less priority is more important
   */
  trait SearchResult[T] extends Response {
    def searchSource: Source
    def result: Results[T]
    def priority: Int
  }

  case class SnippetsResult(searchSource: Source, result: Results[HotelSnippet], priority: Int = 0) extends SearchResult[HotelSnippet]
  case class OffersResult(searchSource: Source, result: Results[Offer], priority: Int = 0) extends SearchResult[Offer]
  case class TourOfferGot(result: Try[ActualizedOffer]) extends Response
  case class SourcesToWait(sourceToWait: Map[Source, Int])

  /*
   * Command
   */
  case class UpdateContext(context: Context)
}
