package ru.yandex.tours.partners

import akka.util.Timeout
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.FutureTry
import spray.http.{StatusCode, StatusCodes, Uri}

import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Try}

/**
  * Created by asoboll on 02.12.16.
  */
trait PartnerRequestProcessing {
  def partner: Partner
  protected def httpClient: AsyncHttpClient

  protected def handle[T](uri: => Try[Uri],
                          parse: String => Try[T],
                          headers: List[(String, String)])
                         (implicit ec: ExecutionContext, timeout: Timeout): Future[T] = {
    for {
      uri <- uri.toFuture
      (sc, response) <- httpClient.get(uri, headers)
      result <- processResponse[T, String](sc, response, parse, uri.toString).toFuture
    } yield result
  }

  protected def handleBytes[T](uri: => Try[Uri],
                               parse: Array[Byte] => Try[T],
                               headers: List[(String, String)])
                              (implicit ec: ExecutionContext, timeout: Timeout): Future[T] = {
    for {
      uri <- uri.toFuture
      (sc, response) <- httpClient.getBytes(uri, headers)
      result <- processResponse[T, Array[Byte]](sc, response, parse, uri.toString).toFuture
    } yield result
  }

  protected def processResponse[T, R](sc: StatusCode,
                                      response: R,
                                      parse: R => Try[T],
                                      uri: String): Try[T] = {
    if (sc != StatusCodes.OK) {
      val entity = response match {
        case s: String => Some(s)
        case bytes: Array[Byte] => Some(scala.io.Source.fromBytes(bytes).getLines().mkString("\n"))
        case _ => None
      }
      val entityAns = entity.fold("Failed to parse response entity")("Entity: " + _)
      Failure(IllegalStatusCodeException(sc, s"$partner responded not 200: $sc. $entityAns. Request uri: $uri"))
    } else {
      parse(response)
    }
  }
}

case class IllegalStatusCodeException(sc: StatusCode, msg: String = "") extends Exception(msg)
