package ru.yandex.tours.partners

import java.util.concurrent.atomic.AtomicInteger

import akka.actor.{Actor, ActorRef}
import com.codahale.metrics.{Histogram, Meter}
import ru.yandex.tours.billing.BillingService
import ru.yandex.tours.direction.Priorities
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.filter.hotel.{GeoIdFilter, PartnerIdFilter}
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners._
import ru.yandex.tours.model.search.{BaseRequest, ExtendedHotelSearchRequest, HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.model.util.Paging
import ru.yandex.tours.operators.SearchSources
import ru.yandex.tours.partners.PartnerProtocol._
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.{Logging, Metrics}

import scala.reflect.ClassTag
import scala.util.{Failure, Success}

class SingleSourceSearcher[S <: Source : ClassTag](partner: Partner,
                                                   client: SourceClient[S],
                                                   searchSources: SearchSources[S],
                                                   geoMapping: GeoMappingHolder,
                                                   hotelsIndex: HotelsIndex,
                                                   billingService: BillingService,
                                                   tree: Tree,
                                                   resortPriorities: Priorities,
                                                   searchSettings: SearchSettingsHolder,
                                                   searchBySpan: Boolean = false,
                                                   isBillingImportant: Boolean = false,
                                                   countrySearch: Boolean = false,
                                                   isDepartureImportant: Boolean = false,
                                                   useSmartFallBack: Boolean = true,
                                                   useSearchByHotelIds: Boolean = true) extends Actor with Logging {

  import context.dispatcher

  private val metrics = Metrics(s"partner.search.result.$partner")

  private val failed = metrics.getMeter("fail", "fail")
  private val skipped = metrics.getMeter("fail", "skip")
  private val nothingFound = metrics.getMeter("success", "nothing-found")
  private val successCommon = metrics.getMeter("success", "found", "by-region")
  private val successBySubregion = metrics.getMeter("success", "found", "by-subregion")
  private val successBySuperregion = metrics.getMeter("success", "found", "by-superregion")
  private val successByHotelInRegion = metrics.getMeter("success", "found", "by-hotels-in-region")
  private val successByHotelInSpan = metrics.getMeter("success", "found", "by-hotels-in-span")
  private val successByHotelPostSearch = metrics.getMeter("success", "found", "by-hotels-post-search")

  private val successCommonCounter = metrics.getHistogram("found", "by-region")
  private val successBySubregionCounter = metrics.getHistogram("found", "by-subregion")
  private val successBySuperregionCounter = metrics.getHistogram("found", "by-superregion")
  private val successByHotelInRegionCounter = metrics.getHistogram("found", "by-hotels-in-region")
  private val successByHotelInSpanCounter = metrics.getHistogram("found", "by-hotels-in-span")
  private val successByHotelPostSearchCounter = metrics.getHistogram("found", "by-hotels-post-search")

  private def processSources[T](sources: Set[Source],
                                _sender: ActorRef,
                                result: (Source, Results[T], Int) => SearchResult[T],
                                action: S => Unit) = {
    _sender ! SourcesToWait(sources.map(_ -> 1).toMap)
    filterSources(sources, _sender, result).foreach(action)
  }

  override def receive: Receive = {
    case SearchHotelsByIds(_, hotelIds, sources, _) if hotelIds.isEmpty || !useSmartFallBack || !useSearchByHotelIds =>
      sender() ! SourcesToWait(sources.map(_ -> 1).toMap)
      sources.foreach { source =>
        sender() ! SnippetsResult(source, Skipped)
      }
    case SearchHotelsByIds(request, hotelIds, sources, _) =>
      val _sender = sender()

      processSources(sources, _sender, SnippetsResult.apply, source => {
        if (isBillingImportant && !billingService.isActive(source.code)) {
          log.info(s"Partner $partner is skipped because is not active in billing. Request $request")
          skipped.mark()
          _sender ! SnippetsResult(source, Skipped)
        } else if (isDepartureImportant && geoMapping.getPartnerDeparture(partner, request.from).isEmpty) {
          log.info(s"Partner $partner doesn't know about departure. Request $request")
          skipped.mark()
          _sender ! SnippetsResult(source, Skipped)
        } else if (shouldSkipLtDuplicatedRequests(request, source)) {
          _sender ! SnippetsResult(source, Skipped)
        } else {
          log.info(s"Partner $partner process post-search by hotel list.. ${hotelIds.mkString("[", ",", "]")}")
          searchByHotels(request.extend(hotelIds = Some(hotelIds)), source, _sender,
            successByHotelPostSearch, successByHotelPostSearchCounter)
        }
      })
    case SearchHotels(request, sources, _) =>
      val _sender = sender()

      processSources(sources, _sender, SnippetsResult.apply, source => {
        if (isBillingImportant && !billingService.isActive(source.code)) {
          log.info(s"Partner $partner is skipped because is not active in billing. Request $request")
          skipped.mark()
          _sender ! SnippetsResult(source, Skipped)
        } else if (isDepartureImportant && geoMapping.getPartnerDeparture(partner, request.from).isEmpty) {
          log.info(s"Partner $partner doesn't know about departure. Request $request")
          skipped.mark()
          _sender ! SnippetsResult(source, Skipped)
        } else if (shouldSkipLtDuplicatedRequests(request, source)) {
          _sender ! SnippetsResult(source, Skipped)
        } else if (!countrySearch && tree.region(request.to).exists(_.isCountry)) {
          log.info(s"Partner $partner could not search by countries. Request $request")
          if (useSmartFallBack) {
            smartFallback(request, source, _sender)
          } else {
            skipped.mark()
            _sender ! SnippetsResult(source, Skipped)
          }
        } else if (!searchBySpan && geoMapping.getPartnerDestination(partner, request.to).isEmpty) {
          log.info(s"Partner $partner unknown destination. Request $request")
          if (useSmartFallBack) {
            smartFallback(request, source, _sender)
          } else {
            skipped.mark()
            _sender ! SnippetsResult(source, Skipped)
          }
        } else {
          client.searchHotels(request.extend(), source) onComplete {
            case Success(snippets) if snippets.isEmpty && useSmartFallBack =>
              log.info(s"Partner $partner finished hotel search, but nothing found. Will try smartFallback. " +
                s"Request $request")
              smartFallback(request, source, _sender)
            case Success(snippets) =>
              log.info(s"Partner $partner finished hotel search. Request $request")
              successCommon.mark()
              successCommonCounter.update(snippets.size)
              _sender ! SnippetsResult(source, Successful(snippets))
            case Failure(e) =>
              failed.mark()
              _sender ! SnippetsResult(source, Failed(e))
          }
        }
      })

    case SearchOffers(request, sources, ctx) =>
      val _sender = sender()
      processSources(sources, _sender, OffersResult.apply, source => {
        if (isBillingImportant && !billingService.isActive(source.code)) {
          log.info(s"Partner $partner is skipped because is not active in billing. Regions $request")
          _sender ! OffersResult(source, Skipped)
        } else if (isDepartureImportant && geoMapping.getPartnerDeparture(partner, request.hotelRequest.from).isEmpty) {
          log.info(s"Partner $partner doesn't know about departure. Request $request")
          _sender ! OffersResult(source, Skipped)
        } else if (shouldSkipLtDuplicatedRequests(request, source)) {
          _sender ! OffersResult(source, Skipped)
        } else {
          hotelsIndex.getHotelById(request.hotelId) match {
            case Some(hotel) if hotel.hasPartner(partner) =>
              client.searchOffers(request.extend(), source) onComplete {
                case Success(offers) =>
                  log.info(s"Partner $partner finished offers search. Request $request")
                  _sender ! OffersResult(source, Successful(offers))
                case Failure(e) =>
                  _sender ! OffersResult(source, Failed(e))
              }
            // Ignore unknown hotels
            case _ => _sender ! OffersResult(source, Skipped)
          }
        }

      })

    case ActualizeTourOffer(request, tourOffer) =>
      val _sender = sender()
      searchSources.getById(tourOffer.getOffer.getSource.getOperatorId) match {
        case Some(source) =>
          client.actualizeOffer(tourOffer, source).onComplete(res => _sender ! TourOfferGot(res))
        case None =>
          _sender ! TourOfferGot(Failure(new Exception(s"Can not find source for offer")))
      }
  }

  private def getFirstRegions(regionId: Int, excludeFirst: Boolean = false): List[Option[Int]] = {
    if (geoMapping.getPartnerDestination(partner, regionId).isEmpty || excludeFirst) Option.empty[Int] ::
      tree.children(regionId).map(_.id).toList.flatMap(getFirstRegions(_))
    else Some(regionId) :: Nil
  }

  private def getSuperRegion(regionId: Int, excludeFirst: Boolean = false): Option[Int] =
    if (geoMapping.getPartnerDestination(partner, regionId).isEmpty || excludeFirst) {
      tree.parent(regionId).flatMap(p => getSuperRegion(p.id))
    } else Some(regionId)

  private def getHotelsByGeoIDWithMetric(geoId: Int, partner: Partner,
                                         count: Int = SingleSourceSearcher.MAX_HOTELS_IN_REQUEST) =
    Option(hotelsIndex.getHotels(Paging(pageSize = count), new GeoIdFilter(geoId), new PartnerIdFilter(partner.id)).
      toList).filter(_.nonEmpty).map((_, successByHotelInRegion, successByHotelInRegionCounter))

  private def getHotelsBySpanWithMetric(geoId: Int, partner: Partner,
                                        count: Int = SingleSourceSearcher.MAX_HOTELS_IN_REQUEST) =
    Option(tree.region(geoId).filter(_.boundingBox.nonEmpty).toList.flatMap {r =>
      hotelsIndex.inRectangle(r.boundingBox, count, new PartnerIdFilter(partner.id)).toList
    }).filter(_.nonEmpty).map((_, successByHotelInSpan, successByHotelInSpanCounter))

  private def searchBySubRegions(request: HotelSearchRequest, source: S, _sender: ActorRef): Boolean = {
    val regionPath = getFirstRegions(request.to, excludeFirst = true).flatten.
      filter(getHotelsByGeoIDWithMetric(_, partner, 1).nonEmpty)

    if (regionPath.nonEmpty) {
      val top = regionPath.map { r ⇒
        resortPriorities.getRating(r) → r
      }.sortBy(-_._1).take(SingleSourceSearcher.MAX_SUBREGIONS).map(_._2)
      log.info(s"Partner $partner use smaller regions (top-${SingleSourceSearcher.MAX_SUBREGIONS}).. $regionPath")

      import ru.yandex.tours.util.lang.Futures
      val found = new AtomicInteger(0)
      Futures.successSequence(top.map { id =>
        val extendedRequest = request.extend(subRegionId = Some(id))
        client.searchHotels(extendedRequest, source).map { result =>
          found.addAndGet(result.size)
          _sender ! SnippetsResult(source, Partial(result))
        }
      }).onComplete {
        case Success(p) =>
          val count = found.get()
          if (count > 0) {
            successBySubregion.mark()
            successBySubregionCounter.update(count)
          }
          else nothingFound.mark()
          _sender ! SnippetsResult(source, Successful(Iterable.empty))
        case Failure(e) =>
          // Theoretically this cannot happen with Futures.successSequence
          failed.mark()
          _sender ! SnippetsResult(source, Failed(e))
      }

      true
    } else false
  }

  private def searchByHotels(request: HotelSearchRequest, source: S, _sender: ActorRef): Boolean =
    useSearchByHotelIds && {
      (
        getHotelsByGeoIDWithMetric(request.to, partner) orElse
          getHotelsBySpanWithMetric(request.to, partner)
        ).exists { case (hotels, successByHotel, successByHotelCounter) =>
        val ids = hotels.map(_.id).distinct
        log.info(s"Partner $partner search by hotel list.. ${ids.mkString("[", ",", "]")}")

        searchByHotels(request.extend(hotelIds = Some(ids)), source, _sender, successByHotel, successByHotelCounter)
      }
    }

  private def searchByHotels(request: ExtendedHotelSearchRequest,
                             source: S,
                             _sender: ActorRef,
                             successMeter: Meter,
                             successCounter: Histogram) =
    request match {
      case ExtendedHotelSearchRequest(_, _, Some(hotelIds)) =>
        client.searchHotels(request, source) onComplete {
          case Success(snippets) =>
            log.info(s"Partner $partner finished hotel search. Request $request")
            if (snippets.nonEmpty) {
              successMeter.mark()
              successCounter.update(snippets.size)
            }
            else nothingFound.mark()
            _sender ! SnippetsResult(source, Successful(snippets))
          case Failure(e) =>
            failed.mark()
            _sender ! SnippetsResult(source, Failed(e))
        }
        true
      case _ => false
    }

  private def searchBySuperRegion(request: HotelSearchRequest, source: S, _sender: ActorRef): Boolean = {
    val superRegion = getSuperRegion(request.to)

    // by super-region, but with respect to countrySearch and existing span of target region
    if (superRegion.nonEmpty && tree.region(request.to).exists(_.boundingBox.nonEmpty) &&
      (countrySearch || !tree.region(superRegion.get).exists(_.isCountry))) {
      log.info(s"Partner $partner use super region ${superRegion.get}")
      client.searchHotels(request.extend(subRegionId = superRegion), source) onComplete {
        case Success(snippets) =>
          log.info(s"Partner $partner finished superregion search. Request $request")
          if (snippets.isEmpty) nothingFound.mark()
          else {
            successBySuperregion.mark()
            successBySuperregionCounter.update(snippets.size)
          }
          _sender ! SnippetsResult(source, Successful(snippets))
        case Failure(e) =>
          failed.mark()
          _sender ! SnippetsResult(source, Failed(e))
      }
      true
    } else false
  }

  private def smartFallback(request: HotelSearchRequest, source: S, _sender: ActorRef) = {
    if (
      !searchBySubRegions(request, source, _sender) &&
        !searchByHotels(request, source, _sender) &&
        !searchBySuperRegion(request, source, _sender)
    ) {
      skipped.mark()
      _sender ! SnippetsResult(source, Skipped)
    }
  }

  private def filterSources[T](sources: Set[Source],
                               _sender: ActorRef,
                               result: (Source, Results[T], Int) => SearchResult[T]): Option[S] = {
    sources.flatMap {
      case found: S if found.mappingFor(partner).contains(found.id.toString) => Some(found)
      case source =>
        _sender ! result(source, Failed(new Exception(s"$partner searcher cannot search for $source source")), 0)
        None
    }.headOption
  }

  private def shouldSkipLtDuplicatedRequests(request: HotelSearchRequest, source: S): Boolean = {
    def ltEnabled = searchSources.getByCode(Partners.lt, source.code).nonEmpty
    def ltActive = searchSettings.getRegionSearchSettings(request.hotelRequest.to).ltActive
    if (partner == Partners.lt) {
      if (ltActive.contains(false)) {
        log.info(s"Partner $partner is skipped because region is not active. Request $request")
        skipped.mark()
        return true
      }
    }
    if (Partners.ltDuplicatingPartners.contains(partner)) {
      if (ltEnabled && ltActive.contains(true)) {
        log.info(s"Partner $partner is skipped because it duplicates LT search. Request $request")
        skipped.mark()
        return true
      }
    }
    false
  }

  private def shouldSkipLtDuplicatedRequests(request: OfferSearchRequest, source: S): Boolean = {
    val ltEnabled = hotelsIndex.getHotelById(request.hotelId).exists(_.partnerId(Partners.lt).nonEmpty)
    if (Partners.ltDuplicatingPartners.contains(partner)) {
      if (ltEnabled) {
        log.info(s"Partner $partner is skipped because it duplicates LT search. Request $request")
        skipped.mark()
        return true
      }
    }
    false
  }
}


object SingleSourceSearcher {
  private val MAX_HOTELS_IN_REQUEST = 50
  private val MAX_SUBREGIONS = 5
}