package ru.yandex.tours.partners.booking

import play.api.libs.json._
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.BaseModel._
import ru.yandex.tours.model.search.SearchProducts._
import ru.yandex.tours.model.search._
import ru.yandex.tours.model.util.proto
import ru.yandex.tours.model.{HotelProvider, Languages}
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.partners.BookingHttp
import ru.yandex.tours.partners.PartnerParsingUtil.{ParsingResult, _}
import ru.yandex.tours.util.parsing.IntValue

import scala.util.control.NonFatal
import scala.util.{Failure, Try}

class BookingResponseParser(hotelsIndex: HotelsIndex, pansionUnifier: PansionUnifier) {

  def parseHotelSnippets(request: ExtendedHotelSearchRequest, provider: HotelProvider)
                        (raw: String): Try[ChunkedSnippets] = {
    Try(Json.parse(raw)).flatMap(json => parseHotelSnippets(request, provider, json)).recoverWith {
      case NonFatal(t) => Failure(new RuntimeException(s"Failed to parse booking response: $raw", t))
    }
  }

  def parseOffers(request: ExtendedOfferSearchRequest, provider: HotelProvider)(raw: String): Try[Iterable[Offer]] = {
    Try(Json.parse(raw)).flatMap(json => parseOffers(request, provider, json))
  }

  def parseHotelSnippets(request: ExtendedHotelSearchRequest,
                         provider: HotelProvider, json: JsValue): Try[ChunkedSnippets] = Try {
    json match {
      case obj: JsObject =>
        val chunks = (json \ "chunks").as[Int]
        val results = (json \ "result").as[JsArray].value
        ChunkedSnippets(chunks, translateAndLogErrors(results, partner, request)(parseSnippet(provider)))
      case arr: JsArray =>
        ChunkedSnippets(1, translateAndLogErrors(arr.value, partner, request)(parseSnippet(provider)))
      case _ => sys.error("Unexpected result from booking: " + json)
    }
  }

  def parseOffers(request: ExtendedOfferSearchRequest,
                  provider: HotelProvider, jsArray: JsValue): Try[Iterable[Offer]] = Try {
    val array = jsArray.as[JsArray].value
    if (array.isEmpty) {
      Iterable.empty[Offer]
    } else {
      val json = array.head
      val hotelId = (json \ "hotel_id").as[String]
      val offerParser = parseOffer(hotelId, request.hotelRequest.guestsWithoutInfants, provider, request.hotelId) _
      translateAndLogErrors((json \ "block").as[JsArray].value, partner, request)(offerParser)
    }
  }

  private def parseOffer(hotelId: String, guestCount: Int, provider: HotelProvider, ourHotelId: Int)
                        (json: JsValue, request: ExtendedBaseRequest): Try[ParsingResult[Offer]] = Try {
    val when = request.hotelRequest.when
    val nights = request.hotelRequest.nights

    val blockId = (json \ "block_id").as[String]
    val price = parsePrice(json \ "min_price", "", request.hotelRequest.currency)
    val name = (json \ "name").as[String]
    val freeCancellation = (json \ "refundable").asOpt[Int].contains(1)
    val maxOccupancy = json \ "max_occupancy" match {
      case JsString(IntValue(occ)) => occ
      case JsNumber(num) => num.toInt
      case _ => Int.MaxValue //unknown
    }
    val multiplier =
      if (guestCount <= maxOccupancy) {
        1
      } else if (request.hotelRequest.kidsWithoutInfants == 0 && guestCount % maxOccupancy == 0) {
        guestCount / maxOccupancy
      } else {
        throw new RuntimeException(s"Too small rooms for request $request: max_occupancy $maxOccupancy")
      }

    (hotelsIndex.getHotel(partner, hotelId), getPartnerHotelUrl(hotelId)) match {
      case (Some(hotel), Some(hotelUrl)) =>
        /*
        val url = BookingHttp.bookingLandingPage(
          hotelId,
          guestCount,
          numberOfRooms = multiplier,
          blockId, when, nights,
          Languages.ru
        ).toString()
        */
        val url = BookingHttp.hotelLandingPage(
          hotelUrl,
          request.hotelRequest.getAges,
          when,
          nights,
          request.hotelRequest.lang,
          request.hotelRequest.currency,
          Some(blockId),
          Some(multiplier)
        ).toString()

        def checkMealPlan(name: String) = {
          json \ name match {
            case JsString(IntValue(1)) => true
            case JsNumber(i) if i.intValue() == 1 => true
            case _ => false
          }
        }

        val pansion = if (checkMealPlan("full_board")) {
          Pansion.FB
        } else if (checkMealPlan("half_board")) {
          Pansion.HB
        } else if (checkMealPlan("all_inclusive")) {
          Pansion.AI
        } else if (checkMealPlan("breakfast_included")) {
          Pansion.BB
        } else {
          Pansion.RO
        }
        val offer = OfferBuilder.build(
          blockId,
          ourHotelId,
          OfferBuilder.source(provider, partner),
          when,
          nights,
          pansion,
          name,
          pansion.toString,
          name,
          withTransfer = false,
          withMedicalInsurance = false,
          withFlight = false,
          price * multiplier,
          Seq(OfferBuilder.link(provider, url)),
          Some(url),
          Some(freeCancellation)
        )
        Parsed(offer)
      case _ => UnknownHotelId(hotelId)
    }
  }

  private def parseSnippet(provider: HotelProvider)
                          (json: JsValue, request: ExtendedBaseRequest): Try[ParsingResult[HotelSnippet]] = Try {
    val when = request.hotelRequest.when
    val nights = request.hotelRequest.nights

    val hotelId = (json \ "hotel_id") match {
     case v: JsNumber => v.value.toString
     case s: JsString => s.value
     case _ => sys.error("Unexpected hotel_id from booking: " + json)
    }

    val roomsCount = (json \ "available_rooms").as[Int]
    (hotelsIndex.getHotel(partner, hotelId), getPartnerHotelUrl(hotelId)) match {
      case (Some(hotel), Some(hotelUrl)) =>
        val ourHotelId = hotel.id
        val min = parseHotelInfo("min_total_", json, request.hotelRequest.currency)
        val max = parseHotelInfo("max_total_", json, request.hotelRequest.currency)

        val url = BookingHttp.hotelLandingPage(hotelUrl, request.hotelRequest.getAges, when, nights,
          request.hotelRequest.lang, request.hotelRequest.currency).toString()

        val sample = OfferBuilder.build(
          min.roomId,
          ourHotelId,
          OfferBuilder.source(provider, partner),
          when,
          nights,
          min.pansion,
          min.roomName,
          min.meal,
          min.roomName,
          withTransfer = false,
          withMedicalInsurance = false,
          withFlight = false,
          min.price,
          Seq(OfferBuilder.link(provider, url)),
          Some(url),
          None
        )

        val landingPage = HotelSnippet.LandingPage.newBuilder().setUrl(url).setProviderId(provider.id)

        val builder = HotelSnippet.newBuilder()
          .setHotelId(ourHotelId)
          .setWithFlight(false)
          .setDateMin(proto.fromLocalDate(when))
          .setDateMax(proto.fromLocalDate(when))
          .setNightsMin(nights)
          .setNightsMax(nights)
          .setPriceMin(min.price)
          .setPriceMax(max.price)
          .setOfferCount(roomsCount)
          .setSample(sample)
          .setLandingPage(landingPage)
          .addSource(OfferBuilder.source(provider, partner))

        val pricedPansion = HotelSnippet.PricedPansion.newBuilder().setPansion(min.pansion).setPrice(min.price)
        builder.addPansions(pricedPansion)
        if (min.pansion != max.pansion) {
          builder.addPansions(HotelSnippet.PricedPansion.newBuilder().setPansion(max.pansion).setPrice(max.price))
        }
        val build: HotelSnippet = builder.build
        Parsed(build)
      case _ => UnknownHotelId(hotelId)
    }
  }

  private def parsePrice(json: JsValue, prefix: String, currency: Currency): Int = {
    val otherCurrencyObject = (json \ "other_currency").asOpt[JsObject].toSeq
    val otherCurrencyArray = (json \ "other_currency").asOpt[JsArray].map(_.value).getOrElse(Seq.empty)

    val prices: Seq[Price] = (otherCurrencyObject ++ otherCurrencyArray :+ json).map(parsePrice(prefix))
    prices.find(_.currency == currency.name())
      .getOrElse(throw new Exception(s"Can not find price in $currency")).amount
  }

  private def parseHotelInfo(prefix: String, json: JsValue, currency: Currency) = {
    val roomName = (json \ s"${prefix}price_room_name").as[String]
    val roomId = (json \ s"${prefix}price_room_id").as[String]
    val meal = (json \ s"${prefix}price_mealplan").as[String]
    val pansion = pansionUnifier.unify(meal).getOrElse(Pansion.RO)
    val price = parsePrice(json, prefix, currency)
    HotelInfo(roomName, roomId, price, meal, pansion)
  }

  private def parsePrice(prefix: String)(json: JsValue): Price = {
    val currency = (json \ "currency_code").asOpt[String].getOrElse((json \ "currency").as[String])
    val price = (json \ s"${prefix}price").as[String].toDouble.toInt
    Price(price, currency)
  }

  private def getPartnerHotelUrl(hotelId: String) = {
    for {
      hotel <- hotelsIndex.getHotel(partner, hotelId)
      info <- hotel.partnerIds.find(_.id == hotelId)
    } yield info.url
  }

  private case class HotelInfo(roomName: String, roomId: String, price: Int, meal: String, pansion: Pansion)

  private case class Price(amount: Int, currency: String)

}
