package ru.yandex.tours.partners.booking

import akka.util.Timeout
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.HotelProvider
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search.{ExtendedHotelSearchRequest, ExtendedOfferSearchRequest}
import ru.yandex.tours.partners.{BookingHttp, HotelProviderClient, PartnerConfig, PartnerRequestProcessing}
import ru.yandex.tours.util.http.AsyncHttpClient

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.Try

class DefaultBookingClient(hotelsIndex: HotelsIndex,
                           geoMapping: GeoMappingHolder,
                           tree: Tree,
                           protected val httpClient: AsyncHttpClient,
                           parser: BookingResponseParser, config: PartnerConfig)(implicit val ec: ExecutionContext)
  extends HotelProviderClient with PartnerRequestProcessing {

  private implicit val timeout = Timeout(60.seconds)
  private val FIRST_CHUNK = 1

  override def partner: Partner = Partners.booking

  override def searchHotels(request: ExtendedHotelSearchRequest,
                            source: HotelProvider): Future[Iterable[HotelSnippet]] = {

    def loadChunk(chunk: Int) = {
      val uri = BookingHttp.getHotelsUri(request, hotelsIndex, geoMapping, tree, chunk)
      handle(uri, parser.parseHotelSnippets(request, source), config.getSearchHeaders)
    }

    loadChunk(FIRST_CHUNK).flatMap { result =>
      if (result.chunks == 1) Future.successful(result.snippets)
      else {
        Future.sequence(2 to result.chunks map loadChunk)
          .map(_.flatMap(_.snippets)).map(_ ++ result.snippets)
      }
    }
  }

  override def searchOffers(request: ExtendedOfferSearchRequest, source: HotelProvider): Future[Iterable[Offer]] = {
    val ids = hotelsIndex.getHotelById(request.hotelId).toSeq.flatMap(_.partnerId(partner))
    require(ids.nonEmpty, s"Hotel ${request.hotelId} not known by Booking")

    val futures = ids.map { id =>
      val checkIn = request.hotelRequest.when
      val checkOut = checkIn.plusDays(request.hotelRequest.nights)
      val uri = Try(BookingHttp.blockAvailability(id, checkIn, checkOut,
        request.hotelRequest.lang, request.hotelRequest.currency))
      handle(uri, parser.parseOffers(request, source), config.getSearchHeaders)
    }

    Future.fold(futures)(Seq.empty[Offer]) { _ ++ _}
  }

}
