package ru.yandex.tours.partners.common

import akka.util.Timeout
import ru.yandex.tours.billing.BillingIndex
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.HotelProvider
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search._
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.partners.{CommonHotelProviderHttp, HotelProviderClient, PartnerRequestProcessing}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.Uri

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.Try

/**
  * See https://docs.google.com/document/d/1CCQozh6AFU77QxD72W2x-1YPgw-C4Jrer_yjsXx-Z98 for API details
  */
class CommonHotelProviderClient(hotelsIndex: HotelsIndex,
                                pansionUnifier: PansionUnifier,
                                geoMapping: GeoMappingHolder,
                                protected val httpClient: AsyncHttpClient,
                                val partner: Partner,
                                searchUrl: String,
                                headers: List[(String, String)] = Nil,
                                timeout: Timeout = Timeout(60.seconds),
                                cmHotelBillingIndex: Option[BillingIndex] = None)
                               (implicit val ec: ExecutionContext)
  extends HotelProviderClient with PartnerRequestProcessing with Logging {

  private val parser = new CommonHotelProviderResponseParser(hotelsIndex, pansionUnifier, partner)

  implicit private val implicitTimeout = timeout

  override def searchHotels(request: ExtendedHotelSearchRequest,
                            source: HotelProvider): Future[Iterable[HotelSnippet]] =
    innerSearchRates(request, source).map { offersToSnippets }

  override def searchOffers(request: ExtendedOfferSearchRequest, source: HotelProvider): Future[Iterable[Offer]] =
    innerSearchRates(request, source)

  private def innerSearchRates(baseRequest: ExtendedBaseRequest, provider: HotelProvider): Future[Iterable[Offer]] = {
    handle(
      getUri(baseRequest),
      parser.parse(baseRequest, _, provider),
      headers
    ).map(filterCMBilling)
  }

  private def getUri(request: ExtendedBaseRequest): Try[Uri] = {
    CommonHotelProviderHttp.getSearchUrl(request, searchUrl, partner, hotelsIndex, geoMapping)
  }

  private def filterCMBilling(offers: Iterable[Offer]): Iterable[Offer] = {
    cmHotelBillingIndex match {
      case Some(billingIndex) => billingIndex.filterOffers(partner, offers)
      case None => offers
    }
  }
}
