package ru.yandex.tours.partners.common

import play.api.libs.json._
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.HotelProvider
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchProducts.Offer
import ru.yandex.tours.model.search.{ExtendedBaseRequest, OfferBuilder}
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.partners.PartnerParsingUtil
import ru.yandex.tours.partners.PartnerParsingUtil.{Parsed, ParsedWithDefaultPansion, ParsingResult, UnknownHotelId}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.parsing.JsonParsing._

import scala.util.Try

class CommonHotelProviderResponseParser(hotelsIndex: HotelsIndex,
                                        pansionUnifier: PansionUnifier,
                                        partner: Partner) extends Logging {

  def parse(request: ExtendedBaseRequest, raw: String, provider: HotelProvider): Try[Iterable[Offer]] = Try {
    val json = Json.parse(raw)
    val rateParser = parseRate(provider) _
    PartnerParsingUtil.translateAndLogErrors(
      objects = json.as[JsArray].value,
      partner = partner,
      request = request
    )(rateParser)
  }

  private def parseRate(provider: HotelProvider)
                       (rate: JsValue, request: ExtendedBaseRequest): Try[ParsingResult[Offer]] = Try {
    val partnerHotelId = parseString(rate, "hotel_id")
    hotelsIndex.getHotel(partner, partnerHotelId) match {
      case Some(hotel) =>
        val id = parseOptString(rate, "id").getOrElse(java.util.UUID.randomUUID.toString)
        val roomName = parseString(rate, "room")

        val rawPansion = parseOptString(rate, "meal").getOrElse("")
        val unifiedPansion = pansionUnifier.unify(rawPansion)
        val pansion = unifiedPansion.getOrElse(Pansion.RO)

        val freeCancellation = parseBoolean(rate, "free_cancellation")
        val price = parseDouble(rate, "total_price").toInt
        val url = parseString(rate, "agent_url")

        val maxOccupancy = parseOptInt(rate, "max_occupancy").getOrElse(Int.MaxValue)
          .ensuring(_ > 0, "Non positive max_occupancy")
        val multiplier =
          if (request.hotelRequest.guestsWithoutInfants <= maxOccupancy) {
            1
          } else if (request.hotelRequest.kidsWithoutInfants == 0
            && request.hotelRequest.guestsWithoutInfants % maxOccupancy == 0) {
            request.hotelRequest.guestsWithoutInfants / maxOccupancy
          } else {
            throw new RuntimeException(s"Too small rooms for request $request: max_occupancy $maxOccupancy")
          }

        val purchaseLink = OfferBuilder.link(provider, url)
        val offer = OfferBuilder.build(
          id,
          hotel.id,
          OfferBuilder.source(provider, partner),
          request.hotelRequest.when,
          request.hotelRequest.nights,
          pansion,
          roomName,
          rawPansion = rawPansion,
          originalRoomCode = roomName,
          withTransfer = false,
          withMedicalInsurance = false,
          withFlight = false,
          price = price * multiplier,
          links = Seq(purchaseLink),
          agentBookingUrl = Some(url),
          freeCancellation = Some(freeCancellation)
        )
        unifiedPansion match {
          case Some(_) => Parsed(offer)
          case None => ParsedWithDefaultPansion(offer, rawPansion)
        }
      case None => UnknownHotelId(partnerHotelId)
    }
  }
}