package ru.yandex.tours.partners.common

import akka.util.Timeout
import org.joda.time.LocalDate
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.TourOperator
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search.SearchResults.ActualizedOffer
import ru.yandex.tours.model.search._
import ru.yandex.tours.partners.{PartnerRequestProcessing, SourceClient}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.hotels.HotelSnippetUtil
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.Uri
import spray.http.Uri.Query

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Success, Try}

/**
 * See https://docs.google.com/document/d/1bYvFrE_G_AA7p2cJQZ7w6XHtLRUYApblAD4ffAcp4jc for API details
 */
class CommonTourOperatorClient(searchUrl: Uri,
                               actualizeUrl: Uri,
                               geoMappingHolder: GeoMappingHolder,
                               hotelsIndex: HotelsIndex,
                               protected val httpClient: AsyncHttpClient,
                               parser: CommonTourOperatorResponseParser,
                               val partner: Partner,
                               explicitRegionType: Boolean = false,
                               tree: region.Tree,
                               authHeaders: List[(String, String)] = Nil,
                               timeout: Timeout = Timeout(60.seconds))
                              (implicit ec: ExecutionContext)
  extends SourceClient[TourOperator] with PartnerRequestProcessing with Logging {

  implicit private val to = timeout

  override def searchHotels(request: ExtendedHotelSearchRequest,
                            source: TourOperator): Future[Iterable[HotelSnippet]] = {
    innerSearchOffers(request, source).map {
      offers => HotelSnippetUtil.aggregate(offers)
    }
  }

  override def searchOffers(request: ExtendedOfferSearchRequest, source: TourOperator): Future[Iterable[Offer]] = {
    innerSearchOffers(request, source)
  }

  override def actualizeOffer(offer: ActualizedOffer, source: TourOperator): Future[ActualizedOffer] = {
    val parameterMap = actualizeUrl.query.toMap + ("id" -> offer.getOffer.getExternalId)
    val uri = Success(actualizeUrl.withQuery(Query(parameterMap)))
    handle(uri, parser.parseActualization(offer.getOffer, source), authHeaders)
  }

  private def innerSearchOffers(request: ExtendedBaseRequest, source: TourOperator): Future[Iterable[Offer]] = {
    handle(buildUrl(request), parser.parseOffers(request, source), authHeaders)
      .map(filterWithoutFlight(request))
  }

  private def filterWithoutFlight(request: ExtendedBaseRequest)(offers: Iterable[Offer]) = {
    val (withFlight, withoutFlight) = offers.partition(_.getWithFlight)
    if (withoutFlight.nonEmpty) {
      log.warn(s"Filtered ${withoutFlight.size} offers without flight for request $request to $partner")
    }
    withFlight
  }

  private def buildUrl(baseRequest: ExtendedBaseRequest) = Try {
    val r = baseRequest.hotelRequest
    val from = geoMappingHolder.getPartnerDeparture(partner, r.from)
      .getOrElse(sys.error(s"Unknown $partner departure: ${r.from}"))
    val nightsFrom = r.nightsRange.head
    val nightsTo = r.nightsRange.last
    val (dateFrom, dateTo) = {
      val dFrom = r.dateRange.head
      val dTo = r.dateRange.last
      val dNow = LocalDate.now()
      if (dTo.compareTo(dNow) < 0) {
        sys.error(s"Failed to process request: End date $dTo cannot be smaller than current date $dNow")
      } else if (dFrom.compareTo(dNow) < 0) (dNow, dTo)
      else (dFrom, dTo)
    }
    def getPartnerHotelIds(ids: Seq[Int]) = hotelsIndex.getHotelsById(ids).values.flatMap(_.partnerId(partner))
    val destination = baseRequest match {
      case ExtendedHotelSearchRequest(_, _, Some(ids)) =>
        val partnersIds = getPartnerHotelIds(ids)
        if (partnersIds.isEmpty) sys.error(s"Hotels $ids not known by $partner")
        partnersIds.map(hi => "hotel_id" -> hi).toSeq
      case hsr: ExtendedHotelSearchRequest if explicitRegionType =>
        val toCountry = Seq("to_country" -> tree.country(hsr.to).flatMap { c =>
          geoMappingHolder.getPartnerCountry(partner, c.id)
        }.getOrElse(sys.error(s"Unknown $partner destination: ${hsr.to}")))
        val toCity = if (geoMappingHolder.getPartnerCountry(partner, hsr.to).isEmpty) {
          Seq("to_city" -> geoMappingHolder.getPartnerCity(partner, hsr.to)
            .getOrElse(sys.error(s"Unknown $partner destination: ${hsr.to}")))
        } else Seq.empty[(String, String)]
        toCountry ++ toCity
      case hsr: ExtendedHotelSearchRequest =>
        val destination = geoMappingHolder.getPartnerDestination(partner, hsr.to)
          .getOrElse(sys.error(s"Unknown $partner destination: ${hsr.to}"))
        Seq("to" -> destination)
      case osr: ExtendedOfferSearchRequest =>
        val partnersIds = getPartnerHotelIds(Seq(osr.hotelId))
        if (partnersIds.isEmpty) sys.error(s"Hotel ${osr.hotelId} not known by $partner")
        partnersIds.map(hi => "hotel_id" -> hi).toSeq
    }
    val params = destination ++ Seq(
      "from" -> from,
      "nights_from" -> nightsFrom.toString,
      "nights_to" -> nightsTo.toString,
      "date_from" -> dateFrom.toString,
      "date_to" -> dateTo.toString
    ) ++ r.ages.map(age => "ages" -> age.toString) ++ searchUrl.query

    searchUrl.withQuery(Query(params: _*))
  }
}
