package ru.yandex.tours.partners.common

import org.joda.time.{DateTime, LocalDate, LocalTime}
import play.api.libs.json.{JsArray, JsObject, JsValue, Json}
import ru.yandex.tours.avia.Airports
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchProducts.{Actualization, Offer}
import ru.yandex.tours.model.search.SearchResults.{ActualizedOffer, ResultInfo}
import ru.yandex.tours.model.search.{ExtendedBaseRequest, OfferBuilder}
import ru.yandex.tours.model.{Languages, TourOperator}
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.partners.PartnerParsingUtil
import ru.yandex.tours.partners.PartnerParsingUtil.{Parsed, ParsingResult, UnknownHotelId, UnknownPansion}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.parsing.JsonParsing._

import scala.collection.JavaConversions._
import scala.util.control.NonFatal
import scala.util.{Failure, Success, Try}

class CommonTourOperatorResponseParser(hotelsIndex: HotelsIndex,
                                       pansionUnifier: PansionUnifier,
                                       geoMapping: GeoMappingHolder,
                                       airports: Airports,
                                       partner: Partner,
                                       regionTree: region.Tree) extends Logging {

  def parseOffers(request: ExtendedBaseRequest, operator: TourOperator)(raw: String): Try[Iterable[Offer]] = {
    Try(Json.parse(raw).as[JsArray].value.map(_.as[JsObject])).map { json =>
      PartnerParsingUtil.translateAndLogErrors(json, partner, request)(parseOffer(operator))
    }
  }

  private def parseOffer(operator: TourOperator)
                        (json: JsObject, request: ExtendedBaseRequest): Try[ParsingResult[Offer]] = Try {
    val rawPansion = (json \ "pansion").as[String]
    val partnerHotelId = parseString(json, "hotel_id")
    (hotelsIndex.getHotel(partner, partnerHotelId), pansionUnifier.unify(rawPansion)) match {
      case (Some(hotel), Some(pansion)) =>
        val roomType = parseString(json, "room_type")
        val transfer = parseBoolean(json, "transfer")
        val insurance = parseBoolean(json, "insurance")
        val when = LocalDate.parse(parseString(json, "date"))
        val nights = parseString(json, "nights").toInt
        val retailLink = parseString(json, "retail_link")
        val agentLink = parseString(json, "agent_link")
        val price = parsePrice(json)
        val flight = parseBoolean(json, "flight")
        val id = parseString(json, "id")
        val offer = OfferBuilder.build(
          id,
          hotel.id,
          OfferBuilder.source(operator, partner),
          when,
          nights,
          pansion,
          roomType,
          rawPansion,
          roomType,
          transfer,
          insurance,
          flight,
          price.totalPrice,
          Seq(OfferBuilder.link(operator, retailLink)),
          Some(agentLink)
        )
        Parsed(offer)
      case (None, _) => UnknownHotelId(partnerHotelId)
      case (_, None) => UnknownPansion(rawPansion)
    }
  }

  def parseActualization(offer: Offer, operator: TourOperator)(raw: String): Try[ActualizedOffer] = {
    Try(Json.parse(raw).as[JsObject]).map { json =>
      val prices = parsePrice(json)
      val flights = (json \ "flights").as[JsArray].value.flatMap(parseTrip)
      val fuelCharge = if (flights.isEmpty) 0 else flights.map(_.additionalPrice).min
      val resultBuilder = Actualization.newBuilder()
        .setFuelCharge(fuelCharge)
        .addAllFlights(asJavaIterable(flights.map(_.toProto)))
        .setInfantPrice(prices.infantPrice)
        .setPrice(prices.totalPrice)
        .setVisaPrice(prices.visaPrice)
        .setWithTransfer(offer.getWithTransfer)
      setLinkOverride(json, resultBuilder, operator)
      ActualizedOffer.newBuilder()
        .setCreated(System.currentTimeMillis())
        .setOffer(offer)
        .setActualizationInfo(resultBuilder.build)
        .setResultInfo(ResultInfo.newBuilder().setIsFromLongCache(false))
        .build
    }
  }

  private def setLinkOverride(json: JsObject, resultBuilder: Actualization.Builder, operator: TourOperator): Unit = {
    for {
      flightChanged <- parseOptBoolean(json, "flight_changed")
      _ = resultBuilder.setFlightChanged(flightChanged)
      link <- parseOptString(json, "retail_link")
    } resultBuilder.addLinkOverride(OfferBuilder.link(operator, link))
  }

  private case class Flight(from: FlightPoint, to: FlightPoint, departure: DateTime, arrival: DateTime, flightNumber: String, airline: String)

  private case class FlightPoint(id: Int, code: String, name: String)

  private case class Prices(basePrice: Int,
                            fuelCharge: Int,
                            infantPrice: Int,
                            transferPrice: Int,
                            insurancePrice: Int,
                            visaPrice: Int) {
    // visa price is optional and shouldn't be added to total price
    val totalPrice = basePrice + fuelCharge + infantPrice + transferPrice + insurancePrice
  }

  private case class Route(flights: Seq[Flight]) {
    private def buildFlightPoint(flightIn: Option[Flight], flightOut: Option[Flight]) = {
      val flightPoint = if (flightIn.nonEmpty) flightIn.get.to else flightOut.get.from
      val flightDefault = flightOut.orElse(flightIn).get

      val builder = Actualization.FlightPoint.newBuilder()
        .setGeoId(flightPoint.id)
        .setCompany(flightDefault.airline)
        .setAirport(flightPoint.name)
        .setAirportCode(flightPoint.code)
        .setFlightNumber(flightDefault.flightNumber)
      if (flightIn.nonEmpty) builder.setArrival(flightIn.get.arrival.toString)
      if (flightOut.nonEmpty) builder.setDeparture(flightOut.get.departure.toString)
      builder.build()
    }

    def toProto: Actualization.Route = {
      val points = for {
        (flight1, flight2) <- (Seq(Option.empty) ++ flights.map(Option.apply))
          .zip(flights.map(Option.apply) ++ Seq(Option.empty))
      } yield {
        buildFlightPoint(flight1, flight2)
      }

      Actualization.Route.newBuilder
        .setIsDirect(flights.size == 1)
        .addAllPoint(asJavaIterable(points))
        .build
    }
  }

  private case class Trip(forward: Route, back: Route, additionalPrice: Int) {
    def toProto: Actualization.Flight = {
      Actualization.Flight.newBuilder()
        .setAdditionalPrice(additionalPrice)
        .setBack(back.toProto)
        .setTo(forward.toProto)
        .build
    }
  }

  private def parseTrip(json: JsValue): Option[Trip] = {
    try {
      val trip = json.as[JsObject]
      val addPrice = parseOptInt(trip, "additional_price").getOrElse(0)
      for {
        forward <- parseRoute(trip \ "to" \ "route")
        back <- parseRoute(trip \ "back" \ "route")
      } yield Trip(forward, back, addPrice)
    } catch {
      case NonFatal(e) =>
        log.warn(s"Can not parse $partner actualization", e)
        None
    }
  }

  private def parseRoute(json: JsValue): Option[Route] = {
    json match {
      case JsArray(values) if values.nonEmpty => Some(Route(values.map(_.as[JsObject]).map(parseFlight)))
      case JsArray(values) => None
      case _ => sys.error(s"Array expected during parsing route, but $json found")
    }
  }

  private def parseFlight(json: JsObject) = {
    val from = parseFlightPoint(json, "source")
    val to = parseFlightPoint(json, "destination")
    val departure = DateTime.parse(parseString(json, "departure"))
    val arrival = DateTime.parse(parseString(json, "arrival"))
    val flightNumber = parseString(json, "flight_no")
    val airline = parseString(json, "airline")

    if (partner == Partners.inna) {
      var innaDepOpt = regionTree.getTimeZone(from.id).map(departure.withZoneRetainFields)
      var innaArrOpt = regionTree.getTimeZone(to.id).map(arrival.withZoneRetainFields)
      lazy val durationOpt = parseOptString(json, "duration").map(LocalTime.parse)

      (innaDepOpt, innaArrOpt) match {
        case (Some(innaDep), Some(innaArr)) =>
          if ((innaArr.getMillis - innaDep.getMillis) != durationOpt.fold(0)(_.getMillisOfDay)) {
            log.debug("Inna flight parser: time zones are not consistent with flight duration. " +
              s"departure: $innaDep, arrival: $innaArr, duration: $durationOpt")
          }
        case (Some(innaDep), None) =>
          innaArrOpt = Option(innaDep.plusMillis(durationOpt.get.getMillisOfDay))
        case (None, Some(innaArr)) =>
          innaDepOpt = Option(innaArr.minusMillis(durationOpt.get.getMillisOfDay))
        case (None, None) =>
          sys.error("Inna flight parser: unknown time zones ")
      }

      Flight(from, to, innaDepOpt.get, innaArrOpt.get, flightNumber, airline)
    } else {
      Flight(from, to, departure, arrival, flightNumber, airline)
    }
  }

  private def parseFlightPoint(json: JsObject, prefix: String) = {
    val id = parseString(json, s"${prefix}_id")
    val code = parseString(json, s"${prefix}_code")
    val name = parseString(json, s"${prefix}_airport_name")
    airports.byIata(code) match {
      case Some(airport) if airport.geoId.isDefined =>
        FlightPoint(airport.geoId.get, code, airport.name(Languages.ru))
      case _ =>
        val geoId = geoMapping.getAirport(partner, id)
          .orElse(geoMapping.getGeoId(partner, id))
          .getOrElse(sys.error(s"Unknown partner airport: $id, $code, $name"))
        FlightPoint(geoId, code, name)
    }
  }

  private def parsePrice(json: JsObject) = {
    val basePrice = parseInt(json, "base_price")
    val fuelCharge = parseOptInt(json, "fuel_charge").getOrElse(0)
    val infantPrice = parseOptInt(json, "infant_price").getOrElse(0)
    val transferPrice = parseOptInt(json, "transfer_price").getOrElse(0)
    val insurancePrice = parseOptInt(json, "insurance_price").getOrElse(0)
    val visaPrice = parseOptInt(json, "visa_price").getOrElse(0)
    Prices(basePrice, fuelCharge, infantPrice, transferPrice, insurancePrice, visaPrice)
  }
}
