package ru.yandex.tours.partners.hotelscombined

import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.HotelProvider
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.hotelscombined.{ResponseObjects => HC}
import ru.yandex.tours.model.search.SearchProducts.Offer
import ru.yandex.tours.model.search.{ExtendedBaseRequest, ExtendedHotelSearchRequest, ExtendedOfferSearchRequest, OfferBuilder}
import ru.yandex.tours.operators.HotelProviders
import ru.yandex.tours.partners.PartnerParsingUtil
import ru.yandex.tours.partners.PartnerParsingUtil.{Parsed, ParsingResult, Skipped, UnknownHotelId}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.hotels.HotelSnippetUtil

import scala.collection.JavaConversions._
import scala.collection.mutable
import scala.util.Try

/**
  * Created by asoboll on 02.12.16.
  */
class HotelsCombinedParser(partner: Partner,
                           hotelsIndex: HotelsIndex,
                           hotelProviders: HotelProviders) extends Logging {
  import HotelsCombinedParser._

  //val unknownProviderMonitoring = Monitorings(s"partners.$partner.providers", devOnly = true)
  //  .errorCount("unknown", 0, 1, 1.hours, 1)

  def parseHotels(request: ExtendedHotelSearchRequest,
                  results: Array[Byte]): Try[HotelsCombinedResponse.Hotels] = Try {
    val searchResult = HC.ApiMultipleHotelSearchResult.parseFrom(results)
    val providerMap: Map[Int, HotelProvider] = parseProviders(searchResult.getProvidersList)

    val toParse: Iterable[ParsableRate] = for {
      hotel <- searchResult.getResultsList
      rateRaw <- hotel.getRatesList
    } yield ParsableRate(rateRaw, hotel.getID.toString)

    val offerMap: Map[HotelProvider, Iterable[Offer]] = parseAndGroup(toParse, providerMap, request)
    val snippetMap = offerMap.mapValues(HotelSnippetUtil.aggregate)
    val numberRates = searchResult.getTotalFilteredResults
    HotelsCombinedResponse.Hotels(snippetMap, searchResult.getIsComplete, numberRates)
  }

  def parseOffers(request: ExtendedOfferSearchRequest,
                  results: Array[Byte]): Try[HotelsCombinedResponse.Offers] = Try {
    val searchResult = HC.ApiSingleHotelSearchResult.parseFrom(results)
    val providerMap: Map[Int, HotelProvider] = parseProviders(searchResult.getProvidersList)

    val toParse: Iterable[ParsableRate] = for {
      rateRaw <- searchResult.getResultsList
    } yield ParsableRate(rateRaw, searchResult.getID.toString)

    val offerMap: Map[HotelProvider, Iterable[Offer]] = parseAndGroup(toParse, providerMap, request)
    HotelsCombinedResponse.Offers(offerMap, searchResult.getIsComplete)
  }

  private def parseAndGroup(toParse: Iterable[ParsableRate],
                            providerMap: Map[Int, HotelProvider],
                            request: ExtendedBaseRequest): Map[HotelProvider, Seq[Offer]] = {
    val rateParser = parseRate(providerMap) _
    val parsedRates = PartnerParsingUtil.translateAndLogErrors(
      toParse,
      partner,
      request
    )(rateParser)

    val idGroupedRates = parsedRates.toMultiMap.withDefault(_ => List.empty)
    for {
      (prov, ids) <- providerMap.inverse
    } yield prov -> ids.flatMap(idGroupedRates)
  }

  private def parseProviders(providers: Iterable[HC.ApiProviderDetail]): Map[Int, HotelProvider] = {
    val unknownProviders = mutable.ListBuffer.empty[String]

    val providerMap = providers.zipWithIndex.flatMap { case (providerRaw, i) =>
      val (providerOpt, unknownOpt) = parseProvider(providerRaw)
      unknownProviders ++= unknownOpt
      providerOpt.map(i -> _)
    }.toMap

    if (unknownProviders.nonEmpty) {
      val msg = s"unknown providers [${unknownProviders.mkString(", ")}] in $partner response"
      //unknownProviderMonitoring.error(msg)
      log.warn(msg)
    }

    providerMap
  }

  private def parseProvider(provider: HC.ApiProviderDetail): (Option[HotelProvider], Option[String]) = {
    val rawCode = provider.getCode
    val hp = hotelProviders.getByCode(partner, rawCode)
    val unknown = if (hp.isEmpty && !ignoredProviderCodes.contains(rawCode)) {
      Some(s"$rawCode-${provider.getName}")
    } else None
    (hp, unknown)
  }

  private def parseRate(providerMap: Map[Int, HotelProvider])
                       (rateRaw: ParsableRate,
                        request: ExtendedBaseRequest): Try[ParsingResult[(Int, Offer)]] = Try {
    val ParsableRate(rate, partnerHotelId) = rateRaw
    providerMap.get(rate.getProviderIndex) match {
      case Some(provider) =>

        hotelsIndex.getHotel(partner, partnerHotelId) match {
          case Some(hotel) =>
            val rateId = java.util.UUID.randomUUID.toString
            val roomName = rate.getRoomName
            val inclusions: Iterable[HC.ApiInclusionType] = rate.getInclusionsList
            val totalPrice = rate.getTotalRate.toInt
            val freeCancellation =
              if (rate.hasHasFreeCancellation) Some(rate.getHasFreeCancellation)
              else None
            val url = s"${rate.getBookUri}&splash=false"
            val purchaseLink = Offer.LinkToPurchase.newBuilder()
              .setLink(url)
              .setPartnerName(HCTitle)
              .setBillingId(partner.toString)
              .build()

            val offer = OfferBuilder.build(
              rateId,
              hotel.id,
              OfferBuilder.source(provider, partner),
              request.hotelRequest.when,
              request.hotelRequest.nights,
              pansion = translatePansion(inclusions.toSeq),
              roomName,
              rawPansion = inclusions.mkString(", "),
              originalRoomCode = roomName,
              withTransfer = false,
              withMedicalInsurance = false,
              withFlight = false,
              price = totalPrice,
              links = Seq(purchaseLink),
              agentBookingUrl = Some(url),
              freeCancellation = freeCancellation
            )
            Parsed((rate.getProviderIndex, offer))
          case None => UnknownHotelId(partnerHotelId)
        }
      case None => Skipped
    }
  }

  private def translatePansion(inclusions: Seq[HC.ApiInclusionType]): Pansion = {
    import HC.ApiInclusionType._
    def ai = inclusions.contains(AllInclusive)
    def breakfast = inclusions.contains(Breakfast)
    def lunch = inclusions.contains(Lunch)
    def dinner = inclusions.contains(Dinner)
    def meals = inclusions.contains(Meals)

    if (ai) {
      Pansion.AI
    } else if (meals) {
      Pansion.FB
    } else (breakfast, lunch, dinner) match {
      case (true, false, false) => Pansion.BB
      case (true, true, true) => Pansion.FB
      case (true, _, _) => Pansion.HB
      case (false, _, true) => Pansion.BD
      case _ => Pansion.RO
    }
  }
}

object HotelsCombinedParser {
  case class ParsableRate(rate: HC.ApiSearchResultRate, hotelId: String)

  val HCTitle = "Hotels Combined"

  protected val ignoredProviderCodes: Set[String] = Set(
    "OLO", "OTE", "PRS"
  )
}
