package ru.yandex.tours.partners.hotelscombined

import akka.actor.{Actor, ActorRef}
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.{HotelProvider, Source, TourOperator}
import ru.yandex.tours.partners.HotelsCombinedHttp.{multipleHotelRequest, singleHotelRequest}
import ru.yandex.tours.partners.PartnerConfig
import ru.yandex.tours.partners.PartnerProtocol._
import ru.yandex.tours.partners.hotelscombined.client.HotelsCombinedClient
import ru.yandex.tours.partners.hotelscombined.downloader.AbstractHCDownloader.DownloadRequest
import ru.yandex.tours.partners.hotelscombined.downloader.{HotelsHCDownloader, OffersHCDownloader}

import scala.collection.mutable
import scala.concurrent.duration.{FiniteDuration, _}
import scala.util.Failure

/**
  * Created by asoboll on 02.12.16.
  */
class HotelsCombinedSearcher(client: HotelsCombinedClient,
                             hotelsIndex: HotelsIndex,
                             geoMapping: GeoMappingHolder,
                             id2key: Map[String, String],
                             config: PartnerConfig)
                            (implicit requestTimeout: FiniteDuration = 55.seconds) extends Actor {

  import context.dispatcher

  def partner = client.partner

  private val hotelsDownloader = new HotelsHCDownloader(client, 3.seconds)
  private val offersDownloader = new OffersHCDownloader(client, 1.seconds)

  override def receive: Receive = {
    case SearchHotels(req, sources, ctx) =>
      withProviders(sources, SnippetsResult.apply, sender()) { providers =>
        val request = req.extend()
        val uri = multipleHotelRequest(request, hotelsIndex, geoMapping, config.searchParameters.get.apiKey.get)
        hotelsDownloader.execute(DownloadRequest(uri, request, providers, sender()))
      }

    case SearchHotelsByIds(req, hotelIds, sources, ctx) =>
      withProviders(sources, SnippetsResult.apply, sender()) { providers =>
        val request = req.extend(hotelIds = Some(hotelIds))
        val uri = multipleHotelRequest(request, hotelsIndex, geoMapping, config.searchParameters.get.apiKey.get)
        hotelsDownloader.execute(DownloadRequest(uri, request, providers, sender()))
      }

    case SearchOffers(req, sources, ctx) =>
      withProviders(sources, OffersResult.apply, sender()) { providers =>
        val request = req.extend()
        val uris = singleHotelRequest(request, hotelsIndex, geoMapping, id2key, config.searchParameters.get.apiKey.get)
        uris.foreach { uri =>
          offersDownloader.execute(DownloadRequest(uri, request, providers, sender()))
        }
      }

    case ActualizeTourOffer(request, tourOffer) =>
      sender() ! TourOfferGot(Failure(new Exception(s"$partner searcher cannot actualize offers")))
  }

  private def splitSources(sources: Set[Source]): (Set[HotelProvider], Set[TourOperator]) = {
    val operators = mutable.ListBuffer.empty[TourOperator]
    val providers = sources.flatMap {
      case s: HotelProvider => Some(s)
      case s: TourOperator =>
        operators += s
        None
    }
    (providers, operators.toSet)
  }

  private def withProviders[T](sources: Set[Source],
                               formResult: (Source, Results[T], Int) => SearchResult[T],
                               _sender: ActorRef)
                              (action: Set[HotelProvider] => Unit): Unit = {
    val (providers, operators) = splitSources(sources)

    _sender ! SourcesToWait(operators.map(_ -> 1).toMap)
    operators.foreach { to =>
      val result = Failed(new Exception(s"$partner searcher cannot search tour operator $to"))
      _sender ! formResult(to, result, 0)
    }

    if (providers.nonEmpty) action(providers)
  }
}
