package ru.yandex.tours.partners.hotelscombined.downloader

import java.util.concurrent.TimeoutException
import java.util.{Timer, TimerTask}

import akka.actor.ActorRef
import ru.yandex.tours.model.search.ExtendedBaseRequest
import ru.yandex.tours.model.{HotelProvider, Source}
import ru.yandex.tours.partners.IllegalStatusCodeException
import ru.yandex.tours.partners.PartnerProtocol._
import ru.yandex.tours.partners.hotelscombined.HotelsCombinedResponse
import ru.yandex.tours.partners.hotelscombined.client.HotelsCombinedClient
import ru.yandex.tours.util.Logging
import spray.http.{StatusCodes, Uri}

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future, Promise}
import scala.util.{Failure, Success, Try}

/**
  * Created by asoboll on 27.01.17.
  */
abstract class AbstractHCDownloader
[Snippet, Response <: HotelsCombinedResponse.Response[Snippet], BaseRequest <: ExtendedBaseRequest]
(client: HotelsCombinedClient, pollInterval : FiniteDuration)
(implicit ec: ExecutionContext, requestTimeout: FiniteDuration) extends Logging {
  import AbstractHCDownloader._

  protected def formResult: (Source, Results[Snippet], Int) => SearchResult[Snippet]

  protected def doSearch(request: DownloadRequest[BaseRequest]): Future[Response]

  protected def partner = client.partner

  protected def metrics: HCMetrics

  def execute(request: DownloadRequest[BaseRequest]): Future[AbstractDownloadResult] = metrics.wrap {
    if (request.providers.isEmpty) {
      Future.successful(skipSearch(sys.error("providers should not be empty"), request))
    } else {
      request.requestHolder ! SourcesToWait(request.providers.map(_ -> 1).toMap)
      request.uri match {
        case uri@Success(u) =>
          log.debug(s"$partner search request ${request.request} is being processed with uri: $u")
          val timeout = waitAsync(requestTimeout)
          process(request, () => timeout.isCompleted)
        case Failure(e) =>
          Future.successful(skipSearch(e, request))
      }
    }
  }

  // Sends results to the request holder, returns status & updated request
  protected def manageResponse(dlRequest: DownloadRequest[BaseRequest])
                              (response: Response): (DownloadResult, DownloadRequest[BaseRequest])

  protected def manageThrowable(e: Throwable) = {
    log.debug(s"$partner search attempt failed", e) ; e
  }

  protected def fallbacks(dlRequest: DownloadRequest[BaseRequest]): PartialFunction[Throwable, (AbstractDownloadResult, DownloadRequest[BaseRequest])]

  protected def process(dlRequest: DownloadRequest[BaseRequest],
                        timeout: () => Boolean): Future[AbstractDownloadResult] = {
    val promise = Promise[AbstractDownloadResult]()
    val wait = waitAsync(pollInterval)
    val searchResult = doSearch(dlRequest)
      .transform(manageResponse(dlRequest), manageThrowable)
      .recover(fallbacks(dlRequest))

    searchResult onComplete {
      case Success((res, _)) if res.isComplete =>
        promise.completeWith(searchResult.map(_._1))
      case Success(_) if timeout() =>
        promise.tryFailure(new TimeoutException(""))
      case Failure(e) if timeout() =>
        log.error("Attempts exceeded. Last error = " + e, e)
        dlRequest.providers.foreach(p => dlRequest.requestHolder ! formResult(p, Failed(e), 0))
        promise.tryFailure(e)

      case Success((res: DownloadResult, dlRequestUpdated)) if res.retryNow =>
        promise.completeWith(process(dlRequestUpdated, timeout))
      case Success((_, dlRequestUpdated)) =>
        promise.completeWith(wait.flatMap(_ => process(dlRequestUpdated, timeout)))

      case Failure(e@IllegalStatusCodeException(StatusCodes.BadRequest, msg)) =>
        log.warn(s"Bad request, stopping attempts. $msg")
        dlRequest.providers.foreach(p => dlRequest.requestHolder ! formResult(p, Failed(e), 0))
        promise.tryFailure(e)
      case Failure(_) =>
        promise.completeWith(wait.flatMap(_ => process(dlRequest, timeout)))
    }
    promise.future
  }

  protected def skipSearch(e: Throwable, request: DownloadRequest[BaseRequest]): AbstractDownloadResult = {
    log.info(s"Partner $partner search skipped because '${e.getMessage}'. Request ${request.request}")
    request.providers.foreach(to => request.requestHolder ! formResult(to, Skipped, 0))
    SkippedDownload
  }
}

object AbstractHCDownloader {

  sealed trait AbstractDownloadResult {
    def isComplete: Boolean
  }

  case object SkippedDownload extends AbstractDownloadResult {
    override def isComplete: Boolean = true
  }

  sealed case class DownloadResult(isComplete: Boolean, isEmpty: Boolean, retryNow: Boolean = false)
    extends AbstractDownloadResult

  case class DownloadRequest[+R <: ExtendedBaseRequest](uri: Try[Uri],
                                                        request: R,
                                                        providers: Set[HotelProvider],
                                                        requestHolder: ActorRef)

  private val requestScheduler = new Timer("requestScheduler", true)

  protected def waitAsync(interval: FiniteDuration): Future[Unit] = {
    val sleep = Promise[Unit]()
    requestScheduler.schedule(
      new TimerTask {
        override def run(): Unit = sleep.success(Unit)
      }, interval.toMillis)
    sleep.future
  }
}
