package ru.yandex.tours.partners.hotelscombined.downloader

import java.util.concurrent.TimeoutException

import ru.yandex.common.monitoring.error.ErrorReservoir
import ru.yandex.tours.partners.IllegalStatusCodeException
import ru.yandex.tours.partners.hotelscombined.downloader.AbstractHCDownloader.{AbstractDownloadResult, DownloadResult, SkippedDownload}
import ru.yandex.tours.util.http.HttpMetrics
import ru.yandex.tours.util.{Metrics, Monitorings, ThrottledException}
import spray.http.StatusCodes

import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success}

/**
  * Created by asoboll on 26.12.16.
  */
sealed case class HCMetrics(private val resultMetrics: Metrics,
                            private val httpMetrics: HttpMetrics) {
  private val success = resultMetrics.getMeter("success", "found")
  private val nothingFound = resultMetrics.getMeter("success", "nothing-found")
  private val skipped = resultMetrics.getMeter("fail", "skip")
  private val failed = resultMetrics.getMeter("fail", "fail")
  private val throttled = resultMetrics.getMeter("fail", "throttle")

  def wrap(future: => Future[AbstractDownloadResult])(implicit ec: ExecutionContext): Future[AbstractDownloadResult] = {
    val ctx = httpMetrics.start()
    future.andThen {
      case Success(result: DownloadResult) =>
        ctx.finish(StatusCodes.OK)
        if (result.isEmpty) nothingFound.mark()
        else success.mark()
      case Success(SkippedDownload) =>
        skipped.mark()
      case Failure(exception) =>
        failed.mark()
        exception match {
          case _: TimeoutException => httpMetrics.timeout()
          case _: ThrottledException => throttled.mark()
          case IllegalStatusCodeException(sc, _) => ctx.finish(sc, Some(exception))
          case _ => ctx.finish(StatusCodes.ServiceUnavailable)
        }
    }
  }
}

object HCMetrics {
  def apply(partner: String, component: String): HCMetrics = HCMetrics(
    Metrics(s"partner.search.result.$partner.$component"),
    new HttpMetrics(s"partners.$partner.search.$component", devOnly = true, warningRate = 0.05, maxRate = 0.1, size = 100)
  )
}
