package ru.yandex.tours.partners.leveltravel

import akka.util.Timeout
import ru.yandex.tours.geo.base.region.{Tree, Types}
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.SearchProducts.{Actualization, HotelSnippet, Offer}
import ru.yandex.tours.model.TourOperator
import ru.yandex.tours.model.search.{ExtendedHotelSearchRequest, HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.partners.{LtHttp, PartnerConfig}
import ru.yandex.tours.partners.leveltravel.parsers.{ActualizationParser, HotelSnippetParser, ToursParser}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang._
import spray.http.StatusCode

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.control.NonFatal
import scala.util.{Failure, Success, Try}

/** client for Level.Travel */
class DefaultLtClient(hotelsIndex: HotelsIndex,
                      geoMapping: GeoMappingHolder,
                      tree: Tree,
                      httpClient: AsyncHttpClient,
                      actualizationParser: ActualizationParser,
                      tourOperators: TourOperators,
                      pansionUnifier: PansionUnifier, config: PartnerConfig)
                     (implicit protected val ec: ExecutionContext) extends LtClient with Logging {
  implicit private val timeout = Timeout(10.seconds)
  private val actualizationTimeout = Timeout(30.seconds)
  private val tourParser = new ToursParser(hotelsIndex, tourOperators, pansionUnifier)
  private val snippetsParser = new HotelSnippetParser(hotelsIndex, tourOperators, pansionUnifier)

  def enqueue(request: ExtendedHotelSearchRequest, hotelId: Option[String] = None): Future[EnqueueResponse] = {
    LtHttp.createEnqueueUrl(request, tree, geoMapping, hotelId) match {
      case Success(url) =>
        fetchResource(url, EnqueueResponse.parse(_, tourOperators), "enqueue")
      case Failure(e) => Future.failed(e)
    }
  }

  def status(requestId: String): Future[EnqueueResponse] = {
    fetchResource(LtHttp.status(requestId), EnqueueResponse.parse(_, tourOperators, Some(requestId)), "status")
  }

  def snippets(requestId: String,
               operators: Iterable[TourOperator],
               request: HotelSearchRequest): Future[Iterable[HotelSnippet]] = {
    results(requestId, operators, ids => LtHttp.result(requestId, ids), snippetsParser.parse(requestId, request), "results")
  }

  def tours(requestId: String, hotelId: String,
            operators: Iterable[TourOperator],
            request: OfferSearchRequest): Future[Iterable[Offer]] = {
    results(requestId, operators, ids => LtHttp.tours(hotelId, requestId, ids), tourParser.parse(requestId, hotelId, request), "get_hotel_offers")
  }

  def actualization(requestId: String, tourId: String): Future[Option[Actualization]] = {
    fetchResource(LtHttp.actualization(requestId, tourId), actualizationParser.parse, "actualization")(actualizationTimeout)
  }

  private def results[T](requestId: String, operators: Iterable[TourOperator],
                         idsToUrl: Iterable[String] => String,
                         parser: String => Try[T], name: String) = {
    val ltIds = getLtOperatorIds(operators)
    if (ltIds.isEmpty) {
      Future.failed(new Exception(s"No known operators of LT: $operators"))
    } else {
      fetchResource(idsToUrl(ltIds), parser, name)
    }
  }

  protected def fetch(url: String, requestName: String)(implicit timeout: Timeout): Future[(StatusCode, String)] = {
    httpClient.get(url, config.getSearchHeaders)
  }

  private def fetchResource[T](url: String, parser: String => Try[T], requestName: String)
                              (implicit timeout: Timeout): Future[T] = {
    for {
      (status, entity) <- fetch(url, requestName)
      result <- parseResponse(url, status, entity, parser, requestName).toFuture
    } yield result
  }

  private def parseResponse[T](url: String, sc: StatusCode, entity: String,
                               parser: String => Try[T], requestName: String): Try[T] = {
    if (sc.isSuccess) {
      parser(entity).recover {
        case NonFatal(e) =>
          throw new Exception(s"Can not parse response from `$requestName` request for level travel. " +
            s"Entity: $entity. Url: $url", e)
      }
    } else {
      Failure(new Exception(s"Can not do `$requestName` request for level travel. " +
        s"StatusCode is $sc. Entity: $entity. Url: $url"))
    }
  }

  private def getLtOperatorIds(operators: Iterable[TourOperator]) = {
    def getId(operator: TourOperator) = operator.mappingFor(partner).headOption onEmpty {
      log.warn(s"Unknown tour operator for Level.Travel: $operator")
    }
    operators.flatMap(getId)
  }

}