package ru.yandex.tours.partners.leveltravel

import akka.actor.{Actor, ActorRef, Props}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.{HotelProvider, Source, TourOperator}
import ru.yandex.tours.model.search.{ExtendedHotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.partners.PartnerProtocol._
import ru.yandex.tours.partners.leveltravel.downloader.AbstractDownloader.DownloadRequest
import ru.yandex.tours.partners.leveltravel.downloader.{HotelDownloader, TourDownloader}
import ru.yandex.tours.util.Logging

import scala.util.{Try, Failure, Success}

class LevelTravelSearcher(ltClient: LtClient, hotelsIndex: HotelsIndex, geoMappingHolder: GeoMappingHolder, tree: Tree)
  extends Actor with Logging {

  import context.dispatcher

  private def getSuperRegion(regionId: Int): Option[Int] =
    if (tree.region(regionId).exists(_.isCountry)) {
      None
    } else if (geoMappingHolder.getPartnerDestination(partner, regionId).isEmpty) {
      tree.parent(regionId).flatMap(p => getSuperRegion(p.id))
    } else Some(regionId)

  override def receive: Receive = {
    case SearchHotelsByIds(_, _, sources, _) =>
      sender() ! SourcesToWait(sources.map(_ -> 1).toMap)
      sources.foreach { source =>
        sender() ! SnippetsResult(source, Skipped)
      }
    case SearchHotels(request, sources, ctx) =>
      val _sender = sender()
      _sender ! SourcesToWait(sources.map(_ -> 1).toMap)
      val operators = filterOperators(sources, _sender, hp => SnippetsResult(hp, Failed(new Exception(s"Lt searcher cannot search hotel providers: $hp"))))
      def ltOnComplete(request: ExtendedHotelSearchRequest): PartialFunction[Try[EnqueueResponse], Unit] = {
        case Success(enqueue) =>
          _sender ! UpdateContext(ctx.toBuilder.setLtRequestId(enqueue.requestId).build())
          val downloadRequest = DownloadRequest(enqueue.requestId, operators, _sender)
          log.info(s"Started request_id = ${enqueue.requestId} for request $request (${request.hotelRequest.sessionId})")
          context.actorOf(Props(new HotelDownloader(ltClient, downloadRequest, request.baseRequest)))
        case Failure(e) =>
          log.error("Can not enqueue request to lt", e)
          operators.foreach(operator => _sender ! SnippetsResult(operator, Failed(e)))
      }
      if (geoMappingHolder.getPartnerDeparture(partner, request.from).isEmpty) {
        log.info(s"$request is ignored due to unknown departure")
        operators.foreach(operator => _sender ! SnippetsResult(operator, Skipped))
      } else if (geoMappingHolder.getPartnerDestination(partner, request.to).isEmpty) {
        getSuperRegion(request.to) match {
          case id@Some(superRegionId) if tree.region(request.to).exists(_.boundingBox.nonEmpty) =>
            val extended = request.extend(subRegionId = id)
            log.info(s"Unknown destination for $request, super-region $superRegionId will be used instead")
            ltClient.enqueue(extended) onComplete ltOnComplete(extended)
          case _ =>
            log.info(s"$request is ignored due to unknown destination")
            operators.foreach(operator => _sender ! SnippetsResult(operator, Skipped))
        }
      } else {
        val extended = request.extend()
        ltClient.enqueue(extended) onComplete ltOnComplete(extended)
      }

    case SearchOffers(request, sources, ctx) =>
      val _sender = sender()
      val answerTimes = if (ctx.hasLtRequestId) 2 else 1
      val sourcesToWait: Map[Source, Int] = sources.map {
        case x: HotelProvider => x -> 1
        case x: TourOperator => x -> answerTimes
      }.toMap
      _sender ! SourcesToWait(sourcesToWait)
      val operators = filterOperators(sources, _sender, hp => OffersResult(hp, Failed(new Exception(s"Lt searcher cannot search hotelOpt providers: $hp"))))

      if (geoMappingHolder.getPartnerDeparture(partner, request.hotelRequest.from).isEmpty) {
        log.info(s"$request is ignored due to unknown departure")
        operators.foreach(operator => _sender ! OffersResult(operator, Skipped))
      } else {
        val hotelOpt = hotelsIndex.getHotelById(request.hotelId)
        hotelOpt.flatMap(h => h.partnerId(partner).headOption.map(h -> _)) match {
          case Some((hotel, hotelId)) =>
            if (ctx.hasLtRequestId) {
              createDownloader(ctx.getLtRequestId, operators, _sender, hotelId, request, lowPriority = true)
            }
            val req = request.hotelRequest.copy(to = hotel.geoId).extend()
            val futureRequestId = ltClient.enqueue(req, Some(hotelId)).map(_.requestId).andThen {
              case Success(requestId) =>
                log.info(s"Started request_id = $requestId for request $request (${request.hotelRequest.sessionId})")
            }
            futureRequestId onComplete {
              case Success(requestId) =>
                createDownloader(requestId, operators, _sender, hotelId, request, lowPriority = false)
              case Failure(e) =>
                operators.foreach(operator => _sender ! OffersResult(operator, Failed(e)))
            }
          case None =>
            log.info(s"Skipped request $request (${request.hotelRequest.sessionId}): hotel not mapped to lt")
            for (operator <- operators) {
              for (_ <- 0 until answerTimes) {
                _sender ! OffersResult(operator, Skipped)
              }
            }
        }
      }

    case ActualizeTourOffer(request, tourOffer) =>
      val offer = tourOffer.getOffer
      if (offer.getSource.getPartnerId != partner.id) {
        sender() ! TourOfferGot(Failure(new Exception("Level.Travel can not get ActualizedOffer from not LT Offer")))
      } else if (!offer.getSource.hasRequestId) {
        sender() ! TourOfferGot(Failure(new Exception("Level.Travel Offer doesn't have request_id")))
      } else {
        val requestId = offer.getSource.getRequestId
        val hotelId = request.toursInHotelRequest.hotelId
        hotelsIndex.getHotelById(hotelId) match {
          case Some(hotel) =>
            val res = for {
              optActualization <- ltClient.actualization(requestId, offer.getExternalId)
            } yield {
                optActualization match {
                  case None => tourOffer
                  case Some(ai) =>
                    tourOffer.toBuilder
                      .setCreated(System.currentTimeMillis())
                      .setActualizationInfo(ai)
                      .build()
                }
              }
            val _sender = sender()
            res.onComplete(tryOffer => _sender ! TourOfferGot(tryOffer))
          case None =>
            sender() ! TourOfferGot(Failure(new Exception(s"Unknown hotel id: $hotelId")))
        }
      }
  }

  private def filterOperators(sources: Set[Source], _sender: ActorRef,
                              errorMessage: HotelProvider => Any): Set[TourOperator] = {
    sources.flatMap {
      case x: TourOperator => Some(x)
      case x: HotelProvider =>
        _sender ! errorMessage(x)
        None
    }
  }

  private def createDownloader(requestId: String, operators: Set[TourOperator], _sender: ActorRef,
                               hotelId: String, request: OfferSearchRequest, lowPriority: Boolean) = {
    val downloadRequest = DownloadRequest(requestId, operators, _sender)
    context.actorOf(Props(new TourDownloader(ltClient, downloadRequest, hotelId, request, lowPriority)))
  }
}
