package ru.yandex.tours.partners.leveltravel.downloader

import akka.actor.{Actor, ActorRef}
import ru.yandex.tours.model.RequestStatus._
import ru.yandex.tours.model.search.SearchProducts.Source
import ru.yandex.tours.model.{RequestStatus, TourOperator}
import ru.yandex.tours.partners.PartnerProtocol.{Failed, Results, Skipped, Successful}
import ru.yandex.tours.partners.leveltravel.LtClient
import ru.yandex.tours.partners.leveltravel.downloader.AbstractDownloader.DownloadRequest
import ru.yandex.tours.util.Logging

import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util.{Failure, Success}

object AbstractDownloader {
  case class DownloadRequest(requestId: String, waitingOperators: Set[TourOperator],
                             searcher: ActorRef, attempts: Int = 10)
}

abstract class AbstractDownloader[Snippet, Request, Response](client: LtClient,
                                                              downloadRequest: DownloadRequest,
                                                              searchRequest: Request) extends Actor with Logging {
  import context.dispatcher
  self ! downloadRequest

  override def receive: Receive = {
    case csr@DownloadRequest(requestId, waitingOperators, searcher, attempts) =>
      client.status(requestId) onComplete {
        case Success(enqueueResponse) =>
          val done = waitingOperators.filter(enqueueResponse.isFinished)
          val unknownOperators = waitingOperators.filterNot(enqueueResponse.operators.contains)
          unknownOperators.foreach(op => searcher ! createResponse(op, Skipped))
          processOperators(searcher, enqueueResponse.operators, requestId, done)
          schedule(csr.copy(waitingOperators = waitingOperators diff done diff unknownOperators))
        case Failure(e) =>
          log.warn(s"Can not take status from level.travel. Request id is $requestId")
          schedule(csr, e)
      }
  }

  private def processOperators(searcher: ActorRef, operator2status: Map[TourOperator, RequestStatus],
                               requestId: String, operators: Iterable[TourOperator]) {
    val successfulOperators = operators.flatMap(operator => {
      val status = operator2status(operator)
      if (RequestStatus.isFailed(status)) {
        searcher ! createResponse(operator, Failed(new Exception(s"Operator ${operator.name} failure. Request id is $requestId. Status is $status")))
        None
      } else if (status == RequestStatus.SKIPPED) {
        searcher ! createResponse(operator, Skipped)
        None
      } else {
        Some(operator)
      }
    })
    fetchSnippets(requestId, successfulOperators) onComplete {
      case Success(snippets) =>
        val operatorId2snippets = snippets.groupBy(snippet => getSource(snippet).getOperatorId)
        successfulOperators.foreach(operator => {
          searcher ! createResponse(operator, Successful(operatorId2snippets.getOrElse(operator.id, Iterable.empty)))
        })
      case Failure(e) =>
        val exception = new RuntimeException(s"Failed to fetch LT snippets. Request id is $requestId", e)
        successfulOperators.foreach(operator => failedOperator(searcher, operator, exception))
    }
  }

  private def schedule(req: DownloadRequest, error: Throwable = null) = {
    if (req.attempts > 0) {
      if (req.waitingOperators.nonEmpty) {
        context.system.scheduler.scheduleOnce(3.second, self, req.copy(attempts = req.attempts - 1))
      } else {
        context.stop(self)
      }
    } else {
      if (error ne null) {
        log.error("Attempts exceeded. Last error = " + error, error)
      }
      val exception = new RuntimeException("Attempts exceeded.")
      for (operator <- req.waitingOperators) {
        failedOperator(req.searcher, operator, exception)
      }
      context.stop(self)
    }
  }

  private def failedOperator(searcher: ActorRef, operator: TourOperator, e: Throwable): Unit = {
    searcher ! createResponse(operator, Failed(e))
  }

  protected def getSource(s: Snippet): Source
  
  protected def fetchSnippets(requestId: String, successOperators: Iterable[TourOperator]): Future[Iterable[Snippet]]

  protected def createResponse(operator: TourOperator, results: Results[Snippet]): Response
}
