package ru.yandex.tours.partners.leveltravel.parsers

import org.joda.time.DateTime
import play.api.libs.functional.syntax._
import play.api.libs.json._
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.search.SearchProducts.Actualization
import ru.yandex.tours.model.search.SearchProducts.Actualization._
import ru.yandex.tours.util.Logging

import scala.collection.JavaConversions._
import scala.util.Try
import scala.util.control.NonFatal

class ActualizationParser(geoMappingHolder: GeoMappingHolder) extends Logging {
  def parse(rawJson: String): Try[Option[Actualization]] = Try {
    val json = Json.parse(rawJson)
    if ((json \ "actualized").as[Boolean]) {
      val jsonFlights = (json \ "flights").as[JsArray]
      val roundTrips = jsonFlights.value.map(_.as[JsObject]).flatMap(parseFlight)
      if (roundTrips.isEmpty) sys.error("No valid flights in actualization")

      val minFuelCharge = roundTrips.map(_._3).min
      val flights = roundTrips.map { case (toTrip, backTrip, fuelCharge) =>
        Flight.newBuilder()
          .setTo(toTrip)
          .setBack(backTrip)
          .setAdditionalPrice(fuelCharge - minFuelCharge)
          .build
      }
      val transfer = (json \ "transfer").asOpt[Boolean]
      val extras = (json \ "extras").as[JsArray].value.map(_.as[JsObject])
      val infantPrices = extras.filter(extra => (extra \ "code_name").as[String] == "infant").map(t => (t \ "price").as[Int])
      val infantPrice = if (infantPrices.isEmpty) None else Some(infantPrices.sum)
      val price = (json \ "price").as[Int]
      val visaPrice = (json \ "visa_price").as[Int]
      val tourActualization = Actualization.newBuilder()
        .setVisaPrice(visaPrice)
        .addAllFlights(asJavaIterable(flights))
        .setFuelCharge(minFuelCharge)
        .setPrice(price)
      transfer.foreach(tourActualization.setWithTransfer)
      infantPrice.foreach(tourActualization.setInfantPrice)
      Some(tourActualization.build)
    } else {
      None
    }
  }

  private def parseFlight(flight: JsObject): Option[(Route, Route, Int)] = {
    try {
      val to = parseRoute((flight \ "to").as[JsObject])
      val back = parseRoute((flight \ "back").as[JsObject])
      val fuelCharge = (flight \ "fuel_charge").as[Int]
      Some((to, back, fuelCharge))
    } catch {
      case NonFatal(e) =>
        log.warn(s"Failed to parse flight (${e.getMessage}) from $flight")
        None
    }
  }

  private def parseRoute(x: JsObject): Route = {
    val origin = (x \ "origin").as[Airport]
    val destination = (x \ "destination").as[Airport]
    val departure = validateDateAndGet(x, "departure")
    val arrival = validateDateAndGet(x, "arrival")
    val company = (x \ "airline" \ "name").asOpt[String]
    val flightNumber = (x \ "flight_no").asOpt[String]
    var ans = List(buildFlightPoint(destination, company, None, Some(arrival), None))
    if (x.keys.contains("transition") & x.keys.contains("transition_time")) {
      val transition = (x \ "transition").as[Airport]
      val transitionTime = (x \ "transition_time").as[JsObject]
      val transitionPoint = buildFlightPoint(
        transition,
        company,
        Some(validateDateAndGet(transitionTime, "departure")),
        Some(validateDateAndGet(transitionTime, "arrival")),
        None
      )
      ans = transitionPoint :: ans
    }
    val route = buildFlightPoint(origin, company, Some(departure), None, flightNumber) :: ans
    val isDirect = (x \ "direct").asOpt[Boolean].getOrElse(true) && route.size == 2
    Route.newBuilder()
      .addAllPoint(asJavaIterable(route))
      .setIsDirect(isDirect)
      .build
  }

  private def getGeobaseId(ltCityId: BigDecimal): Option[Int] = {
    if (ltCityId.intValue() >= 16592) None
    else Some(ltCityId.intValue())
  }

  private def getCityId(ltCityId: BigDecimal): Int = {
    geoMappingHolder.getAirport(Partners.lt, ltCityId.toString())
      .orElse(geoMappingHolder.getGeoIdByCity(Partners.lt, ltCityId.toString()))
      .orElse(getGeobaseId(ltCityId))
      .getOrElse(sys.error("No mapping from level.travel city/airport " + ltCityId))
  }

  private def buildFlightPoint(airport: Airport, company: Option[String],
                               departure: Option[String], arrival: Option[String],
                               flightNumber: Option[String]): FlightPoint = {
    val builder = FlightPoint.newBuilder()
      .setGeoId(airport.geoId)
      .setAirportCode(airport.code)
      .setAirport(airport.name)
    departure.foreach(builder.setDeparture)
    arrival.foreach(builder.setArrival)
    company.foreach(builder.setCompany)
    flightNumber.foreach(builder.setFlightNumber)
    builder.build()
  }

  private def validateDateAndGet(x: JsObject, name: String) = {
    val time = (x \ name).as[String]
    DateTime.parse(time)
    time
  }

  case class Airport(geoId: Int, code: String, name: String)

  implicit val airportReads: Reads[Airport] = (
    (__ \ "city" \ "id").read[BigDecimal].map(getCityId) and
    (__ \ "code").read[String] and
    (__ \ "name").read[String]
  )(Airport.apply _)

}
