package ru.yandex.tours.partners.leveltravel.parsers

import org.joda.time.format.DateTimeFormat
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Source}
import ru.yandex.tours.model.util.proto._
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.partners.leveltravel._
import ru.yandex.tours.util.Logging

import scala.collection.JavaConversions._
import scala.collection.mutable
import scala.util.Try
import scala.util.control.NonFatal

/* @author berkut@yandex-team.ru */

class HotelSnippetParser(hotelsIndex: HotelsIndex, tourOperators: TourOperators,
                         pansionUnifier: PansionUnifier) extends Logging {

  private val formatter = DateTimeFormat.forPattern("yyyy-MM-dd")

  def parse(requestId: String, request: HotelSearchRequest)(json: String): Try[Iterable[HotelSnippet]] = Try {
    val obj = new JSONObject(json)
    val array = obj.getJSONArray("result")

    val parsed = Seq.newBuilder[HotelSnippet]
    val warnings = mutable.HashSet.empty[String]
    val errors = mutable.HashMap.empty[JSONObject, Throwable]

    for (i <- 0 until array.length()) {
      val obj = array.getJSONObject(i)
      try {
        for (snippet <- parseSnippet(requestId, obj, warnings, request)) {
          parsed += snippet
        }
      } catch {
        case NonFatal(t) => errors += obj -> t
      }
    }

    if (errors.nonEmpty) {
      log.error(s"${errors.size} invalid snippets for request-id [$requestId]")
      for ((json, error) <- errors) log.error(s"Error in json: $json", error)
    }

    if (warnings.nonEmpty) {
      log.warn(s"${warnings.size} warnings in snippets for request-id [$requestId]: [${warnings.mkString(",")}]")
    }

    parsed.result()
  }

  private def parseSnippet(requestId: String, json: JSONObject, warnings: mutable.HashSet[String],
                           request: HotelSearchRequest): Option[HotelSnippet] = {
    val ltHotelId = json.getInt("hotel_id").toString
    val hotelId = hotelsIndex.getHotel(partner, ltHotelId) match {
      case None =>
        warnings += s"Unknown LT hotel-id: $ltHotelId"
        return Option.empty
      case Some(hotel) =>
        hotel.id
    }
    val minPrice = json.getInt("min_price")
    val maxPrice = json.getInt("max_price")
    val jsonPansion = json.getJSONObject("pansions")

    val pansions = mutable.HashSet.empty[HotelSnippet.PricedPansion]
    jsonPansion.keys().foreach { key =>
      pansionUnifier.unify(key.toString) match {
        case Some(pansion) =>
          pansions += HotelSnippet.PricedPansion.newBuilder()
            .setPansion(pansion)
            .setPrice(jsonPansion.getInt(key.toString))
            .build
        case _ => warnings += s"Unknown pansion: '$key' for hotel $hotelId."
      }
    }
    if (pansions.isEmpty) {
      warnings += s"No known pansions for hotel $hotelId."
      return Option.empty
    }
    val offerCount = json.getInt("offer_count")
    val partnerOperatorId = json.getInt("operator").toString
    val operatorId = tourOperators.getByCode(partner, partnerOperatorId) match {
      case Some(operator) => operator.id
      case None =>
        warnings += "Unknown operator id: " + partnerOperatorId
        return Option.empty
    }
    val nights = json.getInt("nights")
    if (!request.hotelRequest.nightsRange.contains(nights)) {
      throw new Exception(s"Invalid tour. Request is $request, but $nights nights in snippet")
    }
    val startDate = formatter.parseLocalDate(json.getString("start_date"))
    if (!request.hotelRequest.dateRange.contains(startDate)) {
      throw new Exception(s"Invalid tour. Request is $request, but start date is $startDate in snippet")
    }
    val source = Source.newBuilder().setOperatorId(operatorId).setPartnerId(partner.id).setRequestId(requestId)
    Some(
      HotelSnippet.newBuilder()
        .addSource(source)
        .setHotelId(hotelId)
        .setPriceMin(minPrice)
        .setPriceMax(maxPrice)
        .addAllPansions(pansions)
        .setOfferCount(offerCount)
        .setNightsMin(nights)
        .setNightsMax(nights)
        .setDateMin(fromLocalDate(startDate))
        .setDateMax(fromLocalDate(startDate))
        .build
    )
  }

}
