package ru.yandex.tours.partners.leveltravel.parsers

import org.joda.time.LocalDate
import org.joda.time.format.DateTimeFormat
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.TourOperator
import ru.yandex.tours.model.search.SearchProducts.{Offer, Source}
import ru.yandex.tours.model.search.{OfferBuilder, OfferSearchRequest}
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.partners.LtHttp
import ru.yandex.tours.partners.leveltravel._
import ru.yandex.tours.util.Logging

import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.util.Try
import scala.util.control.NonFatal

/* @author berkut@yandex-team.ru */

class ToursParser(hotelsIndex: HotelsIndex, tourOperators: TourOperators, pansionUnifier: PansionUnifier) extends Logging {
  val levelTravelTitle = "Level Travel"
  val dateFormat = DateTimeFormat.forPattern("yyyy-MM-dd")

  def parse(requestId: String, ltHotelId: String, request: OfferSearchRequest)(json: String): Try[Seq[Offer]] = Try {
    val obj = new JSONObject(json)
    val array = obj.getJSONArray("hotel_offers")
    val unknownPansions = new mutable.HashSet[String]

    val parsed = Seq.newBuilder[Offer]
    val errors = mutable.HashMap.empty[JSONObject, Throwable]

    for (i <- 0 until array.length()) {
      val tourJson = array.getJSONObject(i)
      try {
        for (offer <- parseOffers(requestId, ltHotelId, request, tourJson, unknownPansions)) {
          parsed += offer
        }
      } catch {
        case NonFatal(t) => errors += tourJson -> t
      }
    }

    if (errors.nonEmpty) {
      log.error(s"${errors.size} invalid tours for request-id [$requestId]")
      for ((json, error) <- errors) log.error(s"Error in json: $json", error)
    }

    if (unknownPansions.nonEmpty) {
      log.warn(s"Unknown pansions in tours for request-id [$requestId]: ${unknownPansions.mkString(",")}")
    }

    parsed.result()
  }

  private def parseOffers(requestId: String, ltHotelId: String, request: OfferSearchRequest, json: JSONObject, unknownPansions: mutable.HashSet[String]) = {
    val operator = getTourOperator(json.getInt("operator"))
    val nights = json.getInt("nights_count")
    if (!request.hotelRequest.nightsRange.contains(nights)) throw new Exception(s"Invalid tour. Request is $request, but $nights nights in tour")
    val startDate = LocalDate.parse(json.getString("start_date"), dateFormat)
    if (!request.hotelRequest.dateRange.contains(startDate)) throw new Exception(s"Invalid tour. Request is $request, but start date is $startDate in tour")
    val roomType = json.getString("room_type").trim()
    val id = json.getString("id")
    val price = json.getInt("price")
    val originalRoomCode = json.getString("original_room_type")
    val agentUrl = if (json.has("agent_booking_url")) Some(json.getString("agent_booking_url")) else None
    val links = parseLinks(json, requestId, id, operator)
    val hotelId = getHotelId(ltHotelId)

    val source = Source.newBuilder()
      .setOperatorId(operator.id)
      .setPartnerId(partner.id)
      .setRequestId(requestId)
    val rawPansion = json.getString("pansion")
    pansionUnifier.unify(rawPansion) match {
      case Some(pansion) =>
        val offer = OfferBuilder.build(
          id,
          hotelId,
          source.build,
          startDate,
          nights,
          pansion,
          roomType,
          rawPansion,
          originalRoomCode,
          // Inna Tour operator doesn't have transfers
          withTransfer = operator.id != 9,
          withMedicalInsurance = operator.id != 9,
          withFlight = true,
          price,
          links,
          agentUrl,
          Some(false)
        )
        Some(offer)
      case _ =>
        unknownPansions += rawPansion
        None
    }
  }

  private def parseLinks(json: JSONObject, requestId: String, tourId: String, operator: TourOperator) = {
    val links = ListBuffer.empty[Offer.LinkToPurchase]

    links += createLink(levelTravelTitle, LtHttp.tourLink(tourId, requestId), partner.toString)
    val operatorPurchaseUrl = if (json.has("retail_booking_url") && !json.isNull("retail_booking_url") && !json.getString("retail_booking_url").isEmpty) {
      Some(json.getString("retail_booking_url"))
    } else if (json.has("agent_booking_url")) {
      Some(json.getString("agent_booking_url"))
    } else None
    operatorPurchaseUrl.foreach(url => links += createLink(operator.name, url, operator.code))
    links.toList
  }

  private def createLink(name: String, url: String, billingId: String) = Offer.LinkToPurchase.newBuilder()
    .setLink(url)
    .setPartnerName(name)
    .setBillingId(billingId)
    .build()

  private def getTourOperator(id: Int) = {
    tourOperators.getByCode(partner, id.toString) match {
      case None => throw new Exception(s"Unknown tour operator id of Level.Travel: $id")
      case Some(operator) => operator
    }
  }

  private def getHotelId(id: String) = {
    hotelsIndex.getHotel(partner, id) match {
      case None => throw new Exception(s"Unknown Level.Travel hotel id: $id")
      case Some(hotel) => hotel.id
    }
  }
}
