package ru.yandex.tours.partners.oktogo

import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.HotelProvider
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search.{ExtendedHotelSearchRequest, ExtendedOfferSearchRequest}
import ru.yandex.tours.partners.AsyncWsdlUtil._
import ru.yandex.tours.partners.HotelProviderClient
import ru.yandex.tours.util.lang._

import scala.collection.JavaConversions._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.Try

class DefaultOktogoClient(travelApiService: TravelApiServiceSoap,
                          hotelsIndex: HotelsIndex,
                          translator: OktogoTranslator)(implicit ec: ExecutionContext) extends HotelProviderClient {

  override def searchHotels(request: ExtendedHotelSearchRequest,
                            provider: HotelProvider): Future[Iterable[HotelSnippet]] = {
    search(
      translator.toHotelRequest(request),
      response => translator.toHotelSnippets(request, getHotelRS(response), provider)
    )
  }

  override def searchOffers(request: ExtendedOfferSearchRequest, provider: HotelProvider): Future[Iterable[Offer]] = {
    val hotel = hotelsIndex.getHotelById(request.hotelId)
      .getOrElse(throw new Exception(s"Unknown hotel: ${request.hotelId}"))
    val futures = hotel.partnerId(partner).map { id =>
      search(translator.toHotelRequest(request, id), getOffers(provider, request))
    }
    Future.fold(futures)(Seq.empty[Offer])(_ ++ _)
  }

  private def search[T](request: Try[HotelRequest],
                        translate: XmlRequestResponse => Try[Iterable[T]]): Future[Iterable[T]] = {
    for {
      hotelRequest <- request.toFuture
      response <- wrap[XmlRequestResponse](travelApiService.xmlRequestAsync(hotelRequest, _))
      snippets <- translate(response).toFuture
    } yield snippets
  }

  private def getOffers(provider: HotelProvider, request: ExtendedOfferSearchRequest)
                       (response: XmlRequestResponse): Try[Iterable[Offer]] = Try {
    for {
      hotelRs <- getHotelRS(response)
      offer <- translator.toOffers(provider)(request, hotelRs).get
    } yield offer
  }

  private def getHotelRS(response: XmlRequestResponse): Iterable[HotelRS] = {
    for {
      result <- Option(response.getXmlRequestResult).toIterable
      products <- Option(result.getProducts).toIterable
      hotelRS <- products.getHotelRS
    } yield hotelRS
  }

  override def partner: Partner = Partners.oktogo
}
