package ru.yandex.tours.partners.oktogo

import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.BaseModel._
import ru.yandex.tours.model.HotelProvider
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.search.SearchProducts._
import ru.yandex.tours.model.search._
import ru.yandex.tours.model.util.proto
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.partners.PartnerParsingUtil._
import ru.yandex.tours.partners.oktogo.OktogoTranslator._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.Dates._
import spray.http.Uri

import scala.collection.JavaConversions._
import scala.util.Try

object OktogoTranslator {
  val affilateId = "yandextravel-xmlprod@oktogo.ru"
  val password = "kNdWmV1YUvhf9B5"
}

/**
  * Documentation:
  * http://xml.oktogo.ru/doc
  */
class OktogoTranslator(geoMapping: GeoMappingHolder,
                       hotelsIndex: HotelsIndex,
                       pansionUnifier: PansionUnifier) extends Logging {

  def toHotelRequest(request: ExtendedHotelSearchRequest): Try[HotelRequest] = Try {
    request match {
      case ExtendedHotelSearchRequest(_, _, Some(ids)) =>
        val hotels = new ArrayOfInt1()
          .withHotelId(hotelsIndex.getHotelsById(ids).values.flatMap(_.partnerId(partner)).toList.map(Integer.valueOf))
        baseHotelRequest(request, _.setHotels(hotels), HotelRequestMethod.GET_AVAILABILITY_BY_HOTELS_LIST)
      case request: ExtendedHotelSearchRequest =>
        val destination = geoMapping.getPartnerCity(partner, request.to).map(_.toInt)
          .getOrElse(throw new Exception(s"Unknown oktogo destination: ${request.to}"))
        baseHotelRequest(request, _.setDestinationId(destination), HotelRequestMethod.GET_AVAILABILITY_BY_DESTINATIONS)
    }
  }

  def toHotelRequest(request: ExtendedOfferSearchRequest, oktogoHotelId: String): Try[HotelRequest] = Try {
    val partnerHotelId = oktogoHotelId.toInt
    baseHotelRequest(request, _.setHotelId(partnerHotelId), HotelRequestMethod.GET_AVAILABILITY_BY_HOTELS)
  }

  def toHotelSnippets(request: ExtendedHotelSearchRequest, hotelInfo: Iterable[HotelRS], provider: HotelProvider): Try[Iterable[HotelSnippet]] = Try {
    translateAndLogErrors(hotelInfo, partner, request)(toHotelSnippet(provider))
  }

  def toOffers(provider: HotelProvider)(request: ExtendedOfferSearchRequest, hotelInfo: HotelRS): Try[Iterable[Offer]] = Try {
    val hotel = hotelsIndex.getHotelById(request.hotelId).getOrElse(throw new Exception(s"Unknown hotel id in request: $request"))
    val partnerHotelId = hotelInfo.getHotelId
    toOffers(partnerHotelId.toString, hotel, provider)(hotelInfo.getRooms.getRoom, request)
  }.flatten

  private def toOffers(partnerHotelId: String, hotel: Hotel, provider: HotelProvider)
                      (rooms: Iterable[Room], request: ExtendedBaseRequest): Try[Iterable[Offer]] = Try {
    rooms.flatMap(room => roomToOffers(partnerHotelId, hotel, provider)(room, request))
  }

  private def roomToOffers(partnerHotelId: String, hotel: Hotel, provider: HotelProvider)(room: Room, request: ExtendedBaseRequest): Iterable[Offer] = {
    room.getRates.getRate.map {
      rate => roomAndRateToOffer(partnerHotelId, hotel, room, rate, request, provider)
    }
  }

  private def toHotelSnippet(provider: HotelProvider)
                            (hotelInfo: HotelRS, request: ExtendedBaseRequest): Try[ParsingResult[HotelSnippet]] = Try {

    val partnerHotelId = hotelInfo.getHotelId.toString
    hotelsIndex.getHotel(partner, partnerHotelId) match {
      case Some(hotel) =>
        val price = hotelInfo.getPriceFrom.toBigInteger.intValue()

        val offers = toOffers(partnerHotelId, hotel, provider)(hotelInfo.getRooms.getRoom, request)
          .getOrElse(throw new Exception(s"can not parse oktogo offers for hotel: ${hotelInfo.getHotelId}"))

        val url = getLanding(partnerHotelId, hotel, request)
        val landingPage = HotelSnippet.LandingPage.newBuilder().setUrl(url).setProviderId(provider.id)
        val builder = HotelSnippet.newBuilder()
          .setHotelId(hotel.id)
          .setWithFlight(false)
          .setDateMin(proto.fromLocalDate(request.hotelRequest.when))
          .setDateMax(proto.fromLocalDate(request.hotelRequest.when))
          .setNightsMin(request.hotelRequest.nights)
          .setNightsMax(request.hotelRequest.nights)
          .setPriceMin(price)
          .addPansions(HotelSnippet.PricedPansion.newBuilder().setPansion(Pansion.RO).setPrice(price))
          .setPriceMax(price)
          .setOfferCount(1)
          .setLandingPage(landingPage)
          .addSource(OfferBuilder.source(provider, partner))
        if (offers.nonEmpty) {
          // Here is workaround. Offers - is just sample from some providers.
          val sample = offers.minBy(_.getPrice)

          val allPansions = offers.map(o => o.getPansion -> o.getPrice)
          val pansions = allPansions.groupBy(_._1).map { case (pansion, prices) =>
            pansion -> prices.map(_._2).min
          } map {
            case (pansion, minPrice) =>
              HotelSnippet.PricedPansion.newBuilder().setPansion(pansion).setPrice(minPrice).build
          }

          builder.setPriceMax(offers.map(_.getPrice).max)
          builder.clearPansions().addAllPansions(asJavaIterable(pansions))
          builder.setSample(sample)
          builder.setOfferCount(offers.size)
        }
        Parsed(builder.build)
      case None => UnknownHotelId(partnerHotelId)
    }
  }

  private def baseHotelRequest(request: ExtendedBaseRequest, specifySearchParams: HotelSearchParameters => Unit, hotelRequestMethod: HotelRequestMethod) = {
    val searchParams = new HotelSearchParameters()
    searchParams.setCheckInDate(request.hotelRequest.when.toXMLGregorianCalendar)
    searchParams.setCheckOutDate(request.hotelRequest.when.plusDays(request.hotelRequest.nights).toXMLGregorianCalendar)
    searchParams.setCurrency(Currency.RUB)
    val guests = new ArrayOfGuest()
    (0 until request.hotelRequest.adults).foreach { _ =>
      val guest: Guest = new Guest
      guest.setAge(35)
      guests.withGuest(guest)
    }
    request.hotelRequest.kidsAges.foreach { age =>
      val guest = new Guest()
      guest.setAge(age)
      guest.setIsChild(true)
      guests.withGuest(guest)
    }
    val roomInfo = new RoomInfo()
    roomInfo.setGuests(guests)
    searchParams.setRooms(new ArrayOfRoomInfo().withRoomInfo(roomInfo))
    specifySearchParams(searchParams)
    val hotelRequest = new HotelRequest()
    hotelRequest.setAffiliateId(affilateId)
    hotelRequest.setPassword(password)
    hotelRequest.setLanguage(1)
    hotelRequest.setHotelSearchParameters(searchParams)
    hotelRequest.setHotelRequestMethod(hotelRequestMethod)
    hotelRequest
  }



  private def getLanding(partnerHotelId: String, hotel: Hotel, baseRequest: ExtendedBaseRequest) = {
    val partnerInfo = hotel.partnerIds.find(pi => pi.partner == partner && pi.id == partnerHotelId)
      .getOrElse(throw new Exception(s"Can not find partner id $partnerHotelId in hotel with id ${hotel.id} in request $baseRequest"))
    val uri = Uri(partnerInfo.url)
    val request = baseRequest.hotelRequest
    val persons: String = request.adults.toString + (if (request.kidsAges.nonEmpty) "." else "") + request.kidsAges.mkString(".")
    uri.withQuery(
      "in" -> request.when.toString(DATE_FORMAT),
      "out" -> request.when.plusDays(request.nights).toString(DATE_FORMAT),
      "occ" -> persons,
      "utm_source" -> "yandextravel"
    ).withFragment("availabilitySearch").toString()
  }

  private def roomAndRateToOffer(partnerHotelId: String, hotel: Hotel, room: Room, rate: Rate, request: ExtendedBaseRequest, provider: HotelProvider): Offer = {
    val roomName = Option(rate.getRoomName).getOrElse("Неизвестное размещение")
    val roomType = room.getRoomType.value()
    val url = getLanding(partnerHotelId, hotel, request)
    val pansion = pansionUnifier.unify(rate.getMealType.value()).getOrElse(throw new Exception(s"Unknown oktogo mealtype: ${rate.getMealType}"))
    OfferBuilder.build(
      externalId = rate.getRateId,
      hotelId = hotel.id,
      source = OfferBuilder.source(provider, partner),
      when = request.hotelRequest.when,
      nights = request.hotelRequest.nights,
      pansion = pansion,
      roomType = roomName,
      rawPansion = rate.getMealType.value(),
      originalRoomCode = roomType,
      withTransfer = false,
      withMedicalInsurance = false,
      withFlight = false,
      price = rate.getTotalPrice.toBigInteger.intValue(),
      links = Seq(OfferBuilder.link(provider, url)),
      agentBookingUrl = Some(url),
      freeCancellation = Some(!rate.isNonRefundable && !rate.getCancellationPolicyRules.getCancellationRule.isEmpty)
    )
  }

  private val DATE_FORMAT = "dd.MM.yyyy"
}
