package ru.yandex.tours.partners.ostrovokv3

import java.util.UUID

import akka.event.EventStream
import akka.util.Timeout
import org.joda.time.DateTime
import ru.yandex.tours.events.SearchEvent.OstrovokResult
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.HotelProvider
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchResults.OstrovokSearchResult
import ru.yandex.tours.model.search._
import ru.yandex.tours.partners.{OstrovokV3Http, PartnerConfig, PartnerRequestProcessing}
import ru.yandex.tours.util.http.AsyncHttpClient

import scala.concurrent.duration._
import collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.Random

class DefaultOstrovokV3Client(hotelsIndex: HotelsIndex,
                              geoMapping: GeoMappingHolder,
                              tree: Tree,
                              protected val httpClient: AsyncHttpClient,
                              parser: OstrovokV3ResponseParser, config: PartnerConfig, fenceCheckProbability: Double,
                              fencedConfig: PartnerConfig, eventStream: EventStream)
                             (implicit val ec: ExecutionContext)
  extends OstrovokV3Client with PartnerRequestProcessing {

  private lazy val rnd = Random

  implicit private val timeout = Timeout(60.seconds)

  override def partner: Partner = Partners.ostrovokv3

  override def searchRates(hotelSearchRequest: ExtendedHotelSearchRequest,
                           provider: HotelProvider): Future[OstrovokV3Response] = {
    innerSearchRates(hotelSearchRequest, provider)
  }

  override def searchRates(tourSearchRequest: ExtendedOfferSearchRequest,
                           provider: HotelProvider): Future[OstrovokV3Response] = {
    innerSearchRates(tourSearchRequest, provider)
  }

  private def innerSearchRates(baseRequest: ExtendedBaseRequest,
                               provider: HotelProvider): Future[OstrovokV3Response] = {

    val pairId = UUID.randomUUID().toString
    val regular = handle(
      OstrovokV3Http.getSearchUri(baseRequest, hotelsIndex, geoMapping),
      parser.parse(baseRequest, _, provider),
      config.getSearchHeaders
    )

    if (fenceCheckProbability > 0.0 && rnd.nextDouble() <= fenceCheckProbability) {
      val hotelIds = {
        baseRequest match {
          case hsr: ExtendedHotelSearchRequest => hsr.hotelIds.getOrElse(Seq.empty)
          case osr: ExtendedOfferSearchRequest => Seq(osr.hotelId)
          case _ => Seq.empty
        }
      }.map(s => s.toString)

      val fenced = handle(
        OstrovokV3Http.getSearchUri(baseRequest, hotelsIndex, geoMapping),
        parser.parse(baseRequest, _, provider),
        fencedConfig.getSearchHeaders
      )

      def sendResult(result: OstrovokV3Response, fenced: Boolean, hotelIds: Seq[String]): Unit = {
        val osr = OstrovokSearchResult.newBuilder()
          .setIsFenced(fenced)
          .addAllOffers(result.offers.asJava)
          .setPairId(pairId)
          .build()
        val res = OstrovokResult(DateTime.now, baseRequest.hotelRequest, hotelIds, osr)
        eventStream.publish(res)
      }

      fenced.onComplete(f => f.map(result => {
        sendResult(result, fenced = true, hotelIds = hotelIds)
      }))
      regular.onComplete(f => f.map(result => {
        sendResult(result, fenced = false, hotelIds = hotelIds)
      }))
    }
    regular
  }

}
