package ru.yandex.tours.partners.ostrovokv3

import org.joda.time.DateTime
import play.api.libs.json.{JsArray, JsValue, Json}
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.HotelProvider
import ru.yandex.tours.model.search.SearchProducts.Offer
import ru.yandex.tours.model.search.{ExtendedBaseRequest, BaseRequest, OfferBuilder}
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang._

import scala.collection.mutable
import scala.util.{Failure, Success, Try}

class OstrovokV3ResponseParser(hotelsIndex: HotelsIndex, pansionUnifier: PansionUnifier) extends Logging {

  def parse(request: ExtendedBaseRequest, raw: String, provider: HotelProvider): Try[OstrovokV3Response] = {
    Try(Json.parse(raw)).flatMap(json => parse(request, json \ "result", provider))
  }

  private def parse(request: ExtendedBaseRequest, json: JsValue, provider: HotelProvider): Try[OstrovokV3Response] = Try {
    OstrovokV3Response(
      parseHotels(request, (json \ "hotels").as[JsArray], provider)
    )
  }

  private def parseHotels(request: ExtendedBaseRequest, hotels: JsArray, provider: HotelProvider): Seq[Offer] = {
    val unknownPansions = mutable.HashSet.empty[String]
    val errors = mutable.HashMap.empty[JsValue, Throwable]
    val results = for {
      hotelJson <- hotels.value
      id = (hotelJson \ "id").as[String]
      hotel <- hotelsIndex.getHotel(partner, id).toSeq
      rate <- (hotelJson \ "rates").as[JsArray].value
      offer <- parseRate(request, rate, hotel.id, unknownPansions, provider) match {
        case Failure(e) => errors += rate -> e; None
        case Success(offer) => Some(offer)
      }
    } yield offer
    if (errors.nonEmpty) {
      log.error(s"${errors.size} invalid rates for request [$request]")
      for ((json, error) <- errors) log.error(s"Error in json: $json", error)
    }

    if (unknownPansions.nonEmpty) {
      log.warn(s"Unknown pansions in rates for request [$request]: ${unknownPansions.mkString(",")}")
    }
    results
  }

  private def parseRate(request: ExtendedBaseRequest, rate: JsValue, hotelId: Int, unknownPansions: mutable.HashSet[String], provider: HotelProvider): Try[Offer] = Try {
    val id = (rate \ "availability_hash").as[String]
    val roomName = (rate \ "room_name").as[String]
    val pansions = for {
      value <- (rate \ "value_adds").as[JsArray].value
      description = (value \ "description").as[String]
      title = (value \ "title").as[String]
      code = (value \ "code").as[String]
      if code == "has_meal"
      pansion <- pansionUnifier.unify(description) onEmpty {
        unknownPansions += description
      }
    } yield (pansion, title)
    val (pansion, rawPansion) = pansions.headOption.getOrElse((Pansion.RO, "Room only"))
    val freeCancellation = isFreeCancellation((rate \ "cancellation_info" \ "free_cancellation_before").asOpt[String])
    val price = (rate \ "rate_price").as[String].toDouble.toInt
    val url = (rate \ "hotelpage").as[String]
    OfferBuilder.build(
      id,
      hotelId,
      OfferBuilder.source(provider, partner),
      request.hotelRequest.when,
      request.hotelRequest.nights,
      pansion,
      roomName,
      rawPansion,
      roomName,
      withTransfer = false,
      withMedicalInsurance = false,
      withFlight = false,
      price,
      Seq(OfferBuilder.link(provider, url)),
      Some(url),
      Some(freeCancellation)
    )
  }

  private def isFreeCancellation(optDate: Option[String]) = {
    val tomorrow = DateTime.now().plusDays(1)
    (for {
      rawDate <- optDate
      date <- Try(DateTime.parse(rawDate)).toOption
      free = date.isAfter(tomorrow)
    } yield free).getOrElse(false)
  }
}
