package ru.yandex.tours.partners.sunmar

import ru.yandex.tours.partners.AsyncWsdlUtil._

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 20.05.15
 */
class DefaultSunmarClient(searchService: ISearchService, credential: OTICredential)
                         (implicit val ec: ExecutionContext) extends SunmarClient {

  def currencies: Future[Seq[Currency]] = {
    wrap[CurrencyListResponse](searchService.currencyListAsync(credential, _)).map {
      _.getCurrencyListResult.getCurrency.asScala
    }
  }

  def countries: Future[Seq[Country]] = {
    wrap[CountryListResponse] {
      searchService.countryListAsync(credential, _)
    }.map(_.getCountryListResult.getCountry.asScala)
  }

  def toPlaces(from: Int, country: Int) = {
    wrap[FilterToPlaceResponse] {
      searchService.filterToPlaceAsync(credential, from, country, _)
    }.map(_.getFilterToPlaceResult.getToPlace.asScala)
  }

  def hotels(country: Int) = {
    wrap[FilterHotelResponse] {
      searchService.filterHotelAsync(credential, country, _)
    }.map(_.getFilterHotelResult.getHotel.asScala)
  }

  def allToPlaces: Future[Seq[ToPlace]] = {
    def placesSeq(fromAreas: Seq[Area], countries: Seq[Country]) = {
      for {
        from <- fromAreas
        country <- countries
      } yield toPlaces(from.getID, country.getID)
    }
    for {
      fromAreas <- fromAreas
      countries <- countries
      toPlaces <- Future.sequence(placesSeq(fromAreas, countries))
    } yield toPlaces.flatten
  }

  def fromAreas: Future[Seq[Area]] = {
    wrap[FromAreaListResponse] {
      searchService.fromAreaListAsync(credential, _)
    }.map(_.getFromAreaListResult.getArea.asScala)
  }

  def packages(request: PackageListRequest): Future[Seq[ProductList]] = {
    wrap[PackageListResponse] {
      searchService.packageListAsync(credential, request, _)
    }.map(_.getPackageListResult.getProductList.asScala)
  }

  def actualize(productList: ProductList): Future[Seq[ProductListDayDetail]] = {
    wrap[AvailableFlightsResponse] {
      searchService.availableFlightsAsync(credential, productList, _)
    }.map(_.getAvailableFlightsResult.getProductListDayDetail.asScala)
  }
}