package ru.yandex.tours.partners.sunmar

import akka.actor.{ActorRef, Actor}
import ru.yandex.tours.model.search.BaseRequest
import ru.yandex.tours.model.{HotelProvider, Source, TourOperator}
import ru.yandex.tours.partners.PartnerProtocol._
import ru.yandex.tours.util.Logging

import scala.concurrent.ExecutionContext
import scala.concurrent.duration._
import scala.util.{Failure, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.05.15
 */
class SunmarSearcher(sunmarClient: SunmarClient, translator: SunmarTranslator)
                    (implicit ec: ExecutionContext) extends Actor with Logging {
  private val pageSize = 100
  private val timeout = 30.seconds
  private val fakeOperator = TourOperator(3, Map.empty, "Sunmar", "sunmar", 60)

  private def getOperator(operators: Iterable[Source]): Option[TourOperator] = {
    if (operators.size > 1) {
      for (operator <- operators) {
        sender() ! SourcesToWait(Map(operator -> 1))
        sender() ! SnippetsResult(operator, Failed(new RuntimeException("Too many operators")))
      }
      None
    } else if (operators.isEmpty) {
//      Some(fakeOperator)
      None
    } else {
      operators.head match {
        case x: HotelProvider =>
          sender() ! SourcesToWait(Map(x -> 1))
          sender() ! SnippetsResult(x, Failed(new RuntimeException(s"Sunmar searcher cannot search by hotel provider $x")))
          None
        case x: TourOperator =>
          sender() ! SourcesToWait(Map(x -> 1))
          Some(x)
      }
    }
  }

  private def doRequest(originalRequest: BaseRequest,
                        req: PackageListRequest,
                        page: Int = 0)
                       (onComplete: Seq[ProductList] => Boolean)
                       (onFailure: Throwable => Unit): Unit = {
    val start = System.currentTimeMillis()
    val future = sunmarClient.packages(req.withPageSize(pageSize).withStartIndex(page * pageSize))
    future.onComplete {
      case Success(res) =>
        val hotels = res.map(_.getHotel.getID.intValue()).toSet.size
        val elapsed = System.currentTimeMillis() - start
        log.info(s"Got ${res.size} tours in $hotels hotels in $elapsed ms. page = $page for $originalRequest")
        val continue = onComplete(res)
        if (continue) {
          doRequest(originalRequest, req, page + 1)(onComplete)(onFailure)
        }
      case Failure(t) =>
        log.error(s"Failed to search for $originalRequest", t)
        onFailure(t)
    }
  }
  private def sendResponse(_sender: ActorRef, response: SearchResult[_]): Unit = {
    if (response.searchSource != fakeOperator) {
      _sender ! response
    }
  }

  override def receive: Receive = {
    case SearchHotelsByIds(_, _, sources, _) =>
      sender() ! SourcesToWait(sources.map(_ -> 1).toMap)
      sources.foreach { source =>
        sender() ! SnippetsResult(source, Skipped)
      }
    case SearchHotels(request, operators, _) =>
      val _sender = sender()
      val start = System.currentTimeMillis()
      for (operator <- getOperator(operators)) {
        translator.toPackageListRequest(request) match {
          case None =>
            log.info(s"Skipped hotel snippet search $request")
            sendResponse(_sender, SnippetsResult(operator, Skipped))
          case Some(req) =>
            doRequest(request, req) { result =>
              val snippets = translator.toHotelSnippets(result, operator)
              if (result.isEmpty || System.currentTimeMillis() - start > timeout.toMillis) {
                sendResponse(_sender, SnippetsResult(operator, Successful(snippets)))
                false
              } else {
                sendResponse(_sender, SnippetsResult(operator, Partial(snippets)))
                true
              }
            } { t => sendResponse(_sender, SnippetsResult(operator, Failed(t))) }
        }
      }

    case SearchOffers(request, operators, _) =>
      val _sender = sender()
      val start = System.currentTimeMillis()
      for (operator <- getOperator(operators)) {
        translator.toPackageListRequest(request) match {
          case None =>
            log.info(s"Skipped tours search $request")
            sendResponse(_sender, OffersResult(operator, Skipped))
          case Some(req) =>
            doRequest(request, req) { result =>
              if (result.isEmpty || System.currentTimeMillis() - start > timeout.toMillis) {
                sendResponse(_sender, OffersResult(operator, Successful(translator.toTours(request, result, operator))))
                false
              } else {
                sendResponse(_sender, OffersResult(operator, Partial(translator.toTours(request, result, operator))))
                true
              }
            } { t => sendResponse(_sender, OffersResult(operator, Failed(t))) }
        }
      }
    case ActualizeTourOffer(request, tourOffer) =>
      val _sender = sender()
      translator.toProduct(tourOffer) match {
        case Some(product) =>
          for (flights <- sunmarClient.actualize(product)) {
            val actualized = translator.actualize(tourOffer, flights)
            _sender ! TourOfferGot(Success(actualized))
          }
        case None =>
          _sender ! TourOfferGot(Success(tourOffer))
      }
  }
}
