package ru.yandex.tours.partners.sunmar

import java.util.UUID
import javax.xml.bind.JAXBElement

import com.google.protobuf.ByteString
import org.joda.time.LocalTime
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.TourOperator
import ru.yandex.tours.model.hotels.Partners.sunmar
import ru.yandex.tours.model.search.SearchProducts.{Actualization, HotelSnippet, Offer}
import ru.yandex.tours.model.search.SearchResults.ActualizedOffer
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferBuilder, OfferSearchRequest}
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.lang.SideEffectTry
import ru.yandex.tours.util.{GZip, Logging, WsdlUtils, lang}

import scala.collection.JavaConversions._
import scala.collection.mutable.ArrayBuffer
import scala.util.Try
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 26.05.15
 */
class SunmarTranslator(geoMapping: GeoMappingHolder,
                       tree: Tree,
                       hotelsIndex: HotelsIndex,
                       pansionUnifier: PansionUnifier) extends Logging {

  def toPackageListRequest(request: HotelSearchRequest): Option[PackageListRequest] = Try {
    val req = new PackageListRequest
    val kids = request.kids
    require(kids < 4, "Only < 4 kids supported")
    req.setAdult(request.adults)
    req.setChild(kids)
    request.kidsAges.drop(0).headOption.foreach(req.setChild1Age(_))
    request.kidsAges.drop(1).headOption.foreach(req.setChild2Age(_))
    request.kidsAges.drop(2).headOption.foreach(req.setChild3Age(_))

    req.setBeginNight(request.nightsRange.head)
    req.setEndNight(request.nightsRange.last)

    req.setBeginDate(request.dateRange.head.toXMLGregorianCalendar)
    req.setEndDate(request.dateRange.last.toXMLGregorianCalendar)

    val region = tree.region(request.to).getOrElse(sys.error(s"Region not found: ${request.to}"))
    val knownRegion = tree.pathToRoot(region)
      .find(r => geoMapping.isKnownDestination(r.id))
      .getOrElse(sys.error(s"No known destination parent for ${region.id}"))

    val country = tree.country(knownRegion).getOrElse(sys.error(s"No parent of type Country for region ${request.to}"))
    val toCountry = geoMapping.getPartnerCountry(sunmar, country.id).getOrElse(sys.error(s"No mapping for region ${country.id}"))

    val city = geoMapping.getPartnerCity(sunmar, knownRegion.id)
    if (country != knownRegion) require(city.isDefined, s"No city mapping for region ${region.id}")

    val from = geoMapping.getPartnerDeparture(sunmar, request.from).getOrElse(sys.error(s"No mapping for region: ${request.from}"))

    req.setFromArea(from.toInt)
    req.setToCountry(toCountry.toInt)
    city.foreach(req.setToPlace)

    req.setPageSize(100)
    req.setNotShowSTOPSale(true)
    req.setOnlyAvailableFlight(true)
    req.setCurrency(4)

    req
  }.onFailure {
    case NonFatal(t) => log.debug("Failed to convert request", t)
  }.toOption

  def toPackageListRequest(request: OfferSearchRequest): Option[PackageListRequest] = {
    val req = toPackageListRequest(request.hotelRequest)
    for {
      r <- req
      hotel <- hotelsIndex.getHotelById(request.hotelId)
      if hotel.hasPartner(sunmar)
    } {
      val code = hotel.partnerIds.filter(_.partner == sunmar).map(_.id).mkString(",")
      log.debug(s"Sunmar hotel code: [$code] all codes ${hotel.partnerIds.map(_.id)}")
      r.setHotel(code)
      if (r.getToPlace eq null) r.setToPlace("")
    }
    req.filter(_.getHotel ne null)
  }

  def toHotelSnippet(product: ProductList, operator: TourOperator): HotelSnippet = {
    val date = product.getBeginDate.toLocalDate.toMillis
    val source = OfferBuilder.source(operator, sunmar)
    val price = product.getTotalPrice.intValue()

    val builder = HotelSnippet.newBuilder()
    builder
      .setDateMin(date)
      .setDateMax(date)
      .setNightsMin(product.getNight)
      .setNightsMax(product.getNight)
      .setPriceMin(price)
      .setPriceMax(price)
      .setOfferCount(1)
      .addSource(source)

    val sunmarHotelId = product.getHotel.getID.toString
    val hotel = hotelsIndex.getHotel(sunmar, sunmarHotelId)
      .getOrElse(sys.error(s"Unknown sunmar hotel $sunmarHotelId"))

    builder.setHotelId(hotel.id)

    val mealCategory = product.getMeal.getMealCategory
    val rawPansions = Seq(mealCategory.getName, mealCategory.getSname, mealCategory.getLname).filter(_ ne null)
    for (pansion <- pansionUnifier.unify(rawPansions)) {
      builder.addPansionsBuilder()
        .setPansion(pansion)
        .setPrice(price)
    }

    builder.build
  }

  def toHotelSnippets(products: Seq[ProductList], operator: TourOperator): Seq[HotelSnippet] = {
    val warnings = ArrayBuffer.empty[String]
    val snippets = products.flatMap { product =>
      Try(toHotelSnippet(product, operator))
        .onFailure(e => warnings += e.toString)
        .toOption
    }
    if (warnings.nonEmpty) log.warn("Warnings in snippets: " + warnings.distinct.mkString(", "))
    snippets
  }

  def toTour(request: OfferSearchRequest, product: ProductList, operator: TourOperator): Offer = {
    val source = OfferBuilder.source(operator, sunmar)
    val price = product.getTotalPrice.intValue()
    val payload = GZip.compress(WsdlUtils.marshal(new ObjectFactory().createProductList(product)))

    val sunmarHotelId = product.getHotel.getID.toString
    val hotel = hotelsIndex.getHotel(sunmar, sunmarHotelId).getOrElse(sys.error(s"Unknown sunmar hotel $sunmarHotelId"))

    val mealCategory = product.getMeal.getMealCategory
    val rawPansions = Seq(mealCategory.getName, mealCategory.getSname, mealCategory.getLname).filter(_ ne null)
    val pansion = pansionUnifier.unify(rawPansions).getOrElse(sys.error(s"Unknown sunmar pansions: $rawPansions"))

    val agencyLink = lang.optional(product.getAgencyLink ne null, product.getAgencyLink)
    val link = lang.optional(product.getShopLink ne null, OfferBuilder.link(operator, product.getShopLink))

    OfferBuilder.build(
      UUID.randomUUID().toString,
      hotel.id,
      source,
      product.getBeginDate.toLocalDate,
      product.getNight,
      pansion,
      product.getRoom.getName,
      product.getMeal.getName,
      product.getRoom.getLname,
      withTransfer = true,
      withMedicalInsurance = true,
      withFlight = true,
      price,
      link,
      agencyLink,
      Some(false),
      Some(ByteString.copyFrom(payload))
    )
  }

  def toTours(request: OfferSearchRequest, products: Seq[ProductList], operator: TourOperator): Seq[Offer] = {
    val warnings = ArrayBuffer.empty[String]
    val tours = products.flatMap { pl =>
      Try(toTour(request, pl, operator))
        .onFailure(e => warnings += e.toString)
        .toOption
    }
    if (warnings.nonEmpty) log.warn("Warnings in tours: " + warnings.distinct.mkString(", "))
    tours
  }

  def toProduct(tourOffer: ActualizedOffer): Option[ProductList] = {
    if (tourOffer.getOffer.hasPayload) {
      val payload = tourOffer.getOffer.getPayload.toByteArray
      val element: JAXBElement[ProductList] = WsdlUtils.unmarshal(GZip.decompress(payload))
      Some(element.getValue)
    } else {
      None
    }
  }

  def actualize(tourOffer: ActualizedOffer, flights: Seq[ProductListDayDetail]): ActualizedOffer = {
    require(flights.nonEmpty, "No flights in actualization")
    val builder = tourOffer.toBuilder
    val actual = builder.getActualizationInfoBuilder

    val minFuelCharge = flights.flatMap(_.getSeatClassDetail.getDayDetailSeatClass).map(_.getExtraPrice.intValue()).min
    val price = flights.map(_.getProductList.getTotalPrice.intValue()).head

    for (flight <- flights) {
      val fl = actual.addFlightsBuilder()
      val to = fl.getToBuilder.setIsDirect(true)
      val back = fl.getBackBuilder.setIsDirect(true)

      to.addPoint { flightPoint(flight.getFlightDate, isDeparture = true) }
      to.addPoint { flightPoint(flight.getFlightDate, isDeparture = false) }
      back.addPoint { flightPoint(flight.getBackFlightDate, isDeparture = true) }
      back.addPoint { flightPoint(flight.getBackFlightDate, isDeparture = false) }

      val extraPrices =
        flight.getSeatClassDetail
          .getDayDetailSeatClass
          .map(_.getExtraPrice.intValue())

      fl.setAdditionalPrice(extraPrices.min - minFuelCharge)
    }
    actual
      .setPrice(price)
      .setFuelCharge(minFuelCharge)

    builder.setCreated(System.currentTimeMillis())
    builder.build()
  }

  private def flightPoint(flightDate: FlightDate, isDeparture: Boolean) = {
    val flight = flightDate.getFlight
    val terminal = if (isDeparture) flightDate.getFromTerminal else flightDate.getToTerminal

    val areaId = terminal.getAirport.getPlace.getArea.getID
    val airportGeoId = geoMapping.getAirport(sunmar, areaId.toString).getOrElse(sys.error(s"No sunmar mapping for airport: $areaId"))
    val tz = tree.getTimeZone(airportGeoId).orNull
    val builder = Actualization.FlightPoint.newBuilder()
      .setGeoId(airportGeoId)
      .setAirport(terminal.getAirport.getLname)
      .setAirportCode(terminal.getAirport.getSname)
      .setCompany(flight.getFlightSupplier.getLname)

    val date = flightDate.getCheckInDate.toLocalDate
    if (isDeparture) {
      val time = LocalTime.parse(flightDate.getDepTime)
      builder
        .setDeparture(date.plusDays(flightDate.getDepDayDiff).toDateTime(time, tz).toString)
        .setFlightNumber(flight.getFlightCode)
    } else {
      val time = LocalTime.parse(flightDate.getArrTime)
      builder.setArrival(date.plusDays(flightDate.getArrDayDiff).toDateTime(time, tz).toString)
    }

    builder.build()
  }
}
