package ru.yandex.tours.searcher.api

import ru.yandex.tours.model.search.{GetOfferRequest, HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.services._
import ru.yandex.tours.util.spray._
import ru.yandex.tours.util.zoo.SharedValue
import spray.routing.{Directive1, Route}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.02.15
 */

class SearcherHandler(toursSearcher: HotelSearchService,
                      minPriceService: MinPriceService,
                      recommendService: RecommendService,
                      searchDirective: Directive1[HotelSearchRequest]) extends HttpHandler with SearchDirectives {

  private val onlyFromLongCache = parameter('only_from_long_cache.as[Boolean] ? false)
  private val canStartRequest = parameter('can_start_request.as[Boolean] ? true)
  private val hotelIds = intArray("hotel_ids", isEmptyOk = true)

  lazy val route: Route = {
    (path("search") & searchDirective & onlyFromLongCache & canStartRequest) {
      (searchRequest, onlyFromLongCache, canStartRequest) =>
        onSuccess(toursSearcher.search(searchRequest, canStartRequest, onlyFromLongCache)) {
          searchResult => completeProto(searchResult)
        }
    } ~ (path("search" / "hotels") & searchDirective & hotelIds & canStartRequest) {
      (searchRequest, hotelIds, canStartRequest) =>
        onSuccess(toursSearcher.searchHotels(searchRequest, hotelIds, canStartRequest)) {
          searchResults => completeProtoSeq(searchResults)
        }
    } ~ (pathPrefix("hotel" / IntNumber) & searchDirective) { (hotelId, searchRequest) =>
      (pathEndOrSingleSlash & canStartRequest) { canStartRequest =>
        val request = OfferSearchRequest(searchRequest, hotelId)
        onSuccess(toursSearcher.searchHotel(request, canStartRequest)) {
          searchResult => completeProto(searchResult)
        }
      } ~ {
        tourRoute(toursSearcher, searchRequest, hotelId)
      }
    } ~ {
      RemoteMinPriceService.routes(minPriceService, searchDirective)
    } ~ {
      RemoteRecommendService.routes(recommendService)
    }
  }

  private def tourRoute(searcher: HotelSearchService, searchRequest: HotelSearchRequest, hotelId: Int): Route = searcher match {
    case tss: OfferSearchService =>
      pathPrefix("tour" / Segment) { tourId =>
        val toursInHotelRequest = OfferSearchRequest(searchRequest, hotelId)
        val tourRequest = GetOfferRequest(toursInHotelRequest, tourId)
        pathEndOrSingleSlash {
          onSuccess(tss.searchOffer(tourRequest)) {
            searchResult => completeProto(searchResult)
          }
        } ~ path("actualize") {
          onSuccess(tss.actualizeOffer(tourRequest)) {
            searchResult => completeProto(searchResult)
          }
        }
      }
    case hss: HotelSearchService => reject()
  }
}
