package ru.yandex.tours.spark

import java.io.File
import java.net.URLClassLoader

import org.apache.commons.io.filefilter.SuffixFileFilter
import ru.yandex.tours.util.{IO, Logging, Zip}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 24.02.16
 */
object Classpath extends Logging {

  private def classpath(cl: ClassLoader): List[File] = cl match {
    case cl1: URLClassLoader =>
      val jars = cl1.getURLs.toList
        .filter(url => !url.getPath.contains("/jre/lib/") && !url.getPath.endsWith("_rt.jar"))
        .map(url => new File(url.getPath))
        .filter(_.exists())
      jars ++ classpath(cl.getParent)
    case null => Nil
    case _ =>
      classpath(cl.getParent)
  }

  private def packJar(dir: File): File = {
    val destination = IO.newTempFile("spark_pack_", suffix = ".jar")
    log.info(s"Packing [$dir] into [$destination]")
    Zip.pack(dir, destination, new SuffixFileFilter(".class"))
    destination
  }

  lazy val listJars: List[File] = {
    val cp = classpath(getClass.getClassLoader).zipWithIndex
    val (classFolders, jars) = cp.partition(_._1.isDirectory)

    val packed = classFolders.map { case (dir, idx) => packJar(dir) -> idx }

    (jars ++ packed).sortBy(_._2).map(_._1)
  }
}
