package ru.yandex.tours.storage

import java.util.concurrent.ConcurrentHashMap
import java.util.function.BiFunction

import akka.actor.ActorSystem
import ru.yandex.tours.model.search.SearchResults.{SearchProgress, Context, HotelSearchResult, OfferSearchResult}
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest}

import scala.concurrent.{Promise, ExecutionContext, Future}
import scala.concurrent.duration.FiniteDuration

class DelayedToursDao(toursDao: ToursDao, delay: FiniteDuration)
                     (implicit ec: ExecutionContext, as: ActorSystem) extends ToursDao {

  private class DelayedSave[Req, Res](realSave: (Req, Res) => Future[Unit],
                                      getProgress: Res => SearchProgress) {

    val delayed = new ConcurrentHashMap[Req, (Res, Promise[Unit])]

    def save(req: Req, res: Res): Future[Unit] = {
      if (getProgress(res).getOperatorTotalCount == 0) return realSave(req, res)

      val (_, p) = delayed.compute(req, new BiFunction[Req, (Res, Promise[Unit]), (Res, Promise[Unit])] {
        override def apply(t: Req, u: (Res, Promise[Unit])): (Res, Promise[Unit]) = {
          Option(u) match {
            case None =>
              schedule(t)
              (res, Promise[Unit]())
            case Some((_, promise)) =>
              (res, promise)
          }
        }
      })
      p.future
    }

    private def schedule(req: Req) = {
      as.scheduler.scheduleOnce(delay) {
        val (res, promise) = delayed.remove(req)
        promise.completeWith(realSave(req, res))
      }
    }
  }

  private val hotelsDelayedSave =
    new DelayedSave[HotelSearchRequest, HotelSearchResult](toursDao.saveHotelSearchResult, _.getProgress)
  private val toursDelayedSave =
    new DelayedSave[OfferSearchRequest, OfferSearchResult](toursDao.saveOffersSearchResult, _.getProgress)

  def isEmpty: Boolean = hotelsDelayedSave.delayed.isEmpty && toursDelayedSave.delayed.isEmpty

  override def saveContext(request: HotelSearchRequest, context: Context): Future[Unit] = {
    toursDao.saveContext(request, context)
  }

  override def getOffersSearchResult(request: OfferSearchRequest): Future[Option[OfferSearchResult]] = {
    toursDao.getOffersSearchResult(request)
  }

  override def getOffersSearchResults(request: HotelSearchRequest,
                                      hotelIds: Iterable[Int]): Future[Seq[OfferSearchResult]] = {
    toursDao.getOffersSearchResults(request, hotelIds)
  }

  override def getHotelSearchResult(request: HotelSearchRequest): Future[Option[HotelSearchResult]] = {
    toursDao.getHotelSearchResult(request)
  }

  override def getContext(request: HotelSearchRequest): Future[Context] = {
    toursDao.getContext(request)
  }

  override def saveHotelSearchResult(request: HotelSearchRequest, response: HotelSearchResult): Future[Unit] = {
    hotelsDelayedSave.save(request, response)
  }

  override def saveOffersSearchResult(request: OfferSearchRequest, response: OfferSearchResult): Future[Unit] = {
    toursDelayedSave.save(request, response)
  }
}
