package ru.yandex.tours.storage

import com.codahale.metrics.MetricRegistry
import ru.yandex.tours.model.search.SearchResults.{Context, HotelSearchResult, OfferSearchResult}
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.util.lang.Futures._

import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.03.15
 */
class InstrumentedToursDao(metrics: MetricRegistry, dao: ToursDao)(implicit ec: ExecutionContext) extends ToursDao {

  val saveContextTimer = metrics.timer("tours_dao.save_context")
  val getContextTimer = metrics.timer("tours_dao.get_context")

  val saveHotelsTimer = metrics.timer("tours_dao.save_hotels")
  val getHotelsTimer = metrics.timer("tours_dao.get_hotels")

  val saveToursTimer = metrics.timer("tours_dao.save_tours")
  val getToursTimer = metrics.timer("tours_dao.get_tours")

  val getManyToursTimer = metrics.timer("tours_dao.get_many_tours")

  val snippetCount = metrics.histogram("tours_dao.snippet.count")

  override def saveContext(request: HotelSearchRequest, context: Context): Future[Unit] = {
    val ctx = saveContextTimer.time()
    dao.saveContext(request, context).withTimerContext(ctx)
  }

  override def getContext(request: HotelSearchRequest): Future[Context] = {
    val ctx = getContextTimer.time()
    dao.getContext(request).withTimerContext(ctx)
  }

  override def saveHotelSearchResult(request: HotelSearchRequest, response: HotelSearchResult): Future[Unit] = {
    snippetCount.update(response.getHotelSnippetCount)
    val ctx = saveHotelsTimer.time()
    dao.saveHotelSearchResult(request, response).withTimerContext(ctx)
  }

  override def getHotelSearchResult(request: HotelSearchRequest): Future[Option[HotelSearchResult]] = {
    val ctx = getHotelsTimer.time()
    dao.getHotelSearchResult(request).withTimerContext(ctx)
  }

  override def saveOffersSearchResult(request: OfferSearchRequest, response: OfferSearchResult): Future[Unit] = {
    val ctx = saveToursTimer.time()
    dao.saveOffersSearchResult(request, response).withTimerContext(ctx)
  }

  override def getOffersSearchResult(request: OfferSearchRequest): Future[Option[OfferSearchResult]] = {
    val ctx = getToursTimer.time()
    dao.getOffersSearchResult(request).withTimerContext(ctx)
  }

  override def getOffersSearchResults(request: HotelSearchRequest,
                                      hotelIds: Iterable[Int]): Future[Seq[OfferSearchResult]] = {
    val ctx = getManyToursTimer.time()
    dao.getOffersSearchResults(request, hotelIds).withTimerContext(ctx)
  }
}
