package ru.yandex.tours.storage

import ru.yandex.tours.model.BaseModel.Currency
import ru.yandex.tours.model.search.SearchResults._
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest, SearchType}
import ru.yandex.tours.search.SearchUtil
import ru.yandex.tours.services.CalendarPushService
import ru.yandex.tours.storage.minprice.MinPriceStorage
import ru.yandex.tours.storage.serpcache.SerpCacheDao
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.Futures._

import scala.concurrent.{ExecutionContext, Future}
import scala.util.control.NonFatal

/* @author berkut@yandex-team.ru */

class ProxyToursStorage(toursStorage: ToursDao,
                        toursCache: ToursDao,
                        minPriceStorage: MinPriceStorage,
                        searchType: SearchType,
                        calendarPushService: CalendarPushService,
                        serpCacheDao: SerpCacheDao
                       )
                       (implicit ec: ExecutionContext) extends ToursDao with Logging {

  private def markAsFromCache(resultInfo: ResultInfo) = {
    resultInfo.toBuilder.setIsFromLongCache(true).build()
  }

  override def saveHotelSearchResult(request: HotelSearchRequest, response: HotelSearchResult): Future[Unit] = {
    var statisticUpdate = Future.successful(())
    if (response.getProgress.getIsFinished) {
      statisticUpdate = minPriceStorage.update(request, response)
      val pushToCalendar = (searchType match {
        case SearchType.TOURS => calendarPushService.saveTourSnippets(request, response)
        case SearchType.ROOMS => calendarPushService.saveRoomSnippets(request, response)
      }).recover {
        case NonFatal(t) =>
          log.warn("Failed to push to calendar")
          Future.successful(())
      }
      statisticUpdate = (statisticUpdate zip pushToCalendar).toUnit
      if (searchType == SearchType.ROOMS && !SearchUtil.isFailed(response)) {
        if (request.hotelRequest.currency == Currency.RUB)
        {
          serpCacheDao.saveHotelSearchResult(request, response)
        }
        else {
          log.info(s"Currency is ${request.hotelRequest.currency.toString} thus skipping serpCache")
        }
      }
      if (!SearchUtil.isFailed(response)) {
        toursCache.saveHotelSearchResult(request, response.toBuilder.setResultInfo(markAsFromCache(response.getResultInfo)).build())
      }
    }
    for {
      _ <- statisticUpdate
      result <- toursStorage.saveHotelSearchResult(request, response)
    } yield result
  }

  override def getOffersSearchResult(request: OfferSearchRequest): Future[Option[OfferSearchResult]] = toursStorage.getOffersSearchResult(request)

  override def getHotelSearchResult(request: HotelSearchRequest): Future[Option[HotelSearchResult]] = toursStorage.getHotelSearchResult(request)

  override def getOffersSearchResults(request: HotelSearchRequest,
                                      hotelIds: Iterable[Int]): Future[Seq[OfferSearchResult]] = {
    toursStorage.getOffersSearchResults(request, hotelIds)
  }

  override def saveOffersSearchResult(request: OfferSearchRequest, response: OfferSearchResult): Future[Unit] = {
    var statisticUpdate = Future.successful(())
    if (response.getProgress.getIsFinished) {
      statisticUpdate = (searchType match {
        case SearchType.TOURS => calendarPushService.saveTourOffers(request, response)
        case SearchType.ROOMS => calendarPushService.saveRoomOffers(request, response)
      }).recover {
        case NonFatal(t) =>
          log.warn("Failed to push to calendar")
          Future.successful(())
      }
      if (!SearchUtil.isFailed(response)) {
        toursCache.saveOffersSearchResult(request, response.toBuilder.setResultInfo(markAsFromCache(response.getResultInfo)).build())
        if (searchType == SearchType.ROOMS) {
          if (request.hotelRequest.currency == Currency.RUB)
          {
            serpCacheDao.saveOffersSearchResult(request, response)
          }
          else {
            log.info(s"Currency is ${request.hotelRequest.currency.toString} thus skipping serpCache")
          }
        }
      }
    }
    for {
      _ <- statisticUpdate
      result <- toursStorage.saveOffersSearchResult(request, response)
    } yield result
  }

  override def saveContext(request: HotelSearchRequest, context: Context): Future[Unit] = toursStorage.saveContext(request, context)

  override def getContext(request: HotelSearchRequest): Future[Context] = toursStorage.getContext(request)
}
