package ru.yandex.tours.storage

import ru.yandex.tours.app._
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.storage.direction._
import ru.yandex.tours.storage.flight._
import ru.yandex.tours.storage.instrumented.InstrumentedYtByteArrayDao
import ru.yandex.tours.storage.minprice._
import ru.yandex.tours.storage.serpcache.SerpCacheDao
import ru.yandex.tours.storage.transfer.{TransferDao, TransfersTwinDao, YtTransferDao}
import ru.yandex.tours.storage.yt.entities._
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.travel.yt.YtDao

import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.01.15
 */
trait StorageComponents {
  this: Application with AkkaSupport =>
  def minPriceStorage: MinPriceStorage

  def directionPriceDao: DirectionPriceDao
  def directionPriceStorage: DirectionPriceStorage

  def toursStorage: ToursDao

  def toursLongCache: ToursDao

  def hotelsMinPriceStorage: MinPriceStorage

  def hotelsDirectionPriceDao: DirectionPriceDao
  def hotelsDirectionPriceStorage: DirectionPriceStorage

  def hotelsLongCache: ToursDao

  def hotelsStorage: ToursDao

  def flightsStorage: FlightsDao

  def transferStorage: TransferDao
}

trait YtStorageComponents extends StorageComponents {
  this: Application with AkkaSupport with MetricsSupport with YtSupport
    with AviaClientSupport // todo remove
    with CalendarPushServiceSupport =>

  import akkaSystem.dispatcher
  private implicit val akka = akkaSystem

  private val storageTtl = config.getFiniteDuration("tours.storage.ttl")
  private val longCacheTtl = config.getFiniteDuration("tours.cache.ttl")
  private val flightsTtl = config.getFiniteDuration("flights.cache.ttl")
  private val transfersTtl = config.getFiniteDuration("transfers.cache.ttl")

  private lazy val minPriceDao = buildMinPriceDao()
  lazy val minPriceStorage = new MinPriceStorage(minPriceDao)
  lazy val directionPriceDao = buildDirectionPriceDao("direction_price")
  lazy val directionPriceStorage = {
    new DirectionPriceStorage(directionPriceDao, minPriceDao, aviaCacheClient = None, SearchType.TOURS)
  }
  lazy val toursLongCache: ToursDao = buildStorageDao("tours_long_cache", longCacheTtl)
  lazy val toursStorage: ToursDao = {
    val dao = buildStorageDao("tours", storageTtl)
    val instrumentedDao = new InstrumentedToursDao(metricsRegistry, dao)
    val ytDaos = ytSettings.serpCacheClusters.map(cluster => createYtDao(classOf[CacheItem], cluster))
    val serpCacheDao = new SerpCacheDao(ytDaos)
    val proxyStorage = new ProxyToursStorage(instrumentedDao, toursLongCache, minPriceStorage,
      SearchType.TOURS, calendarPushService, serpCacheDao)
    new DelayedToursDao(proxyStorage, 500.millis)
  }

  private lazy val hotelMinPriceDao = buildMinPriceDao("hotels_min_price")
  lazy val hotelsMinPriceStorage = new MinPriceStorage(hotelMinPriceDao)
  lazy val hotelsDirectionPriceDao = buildDirectionPriceDao("hotels_direction_price")
  lazy val hotelsDirectionPriceStorage = {
    new DirectionPriceStorage(hotelsDirectionPriceDao, hotelMinPriceDao,
      aviaCacheClient = Some(aviaCacheClient), searchType = SearchType.ROOMS)
  }
  lazy val hotelsLongCache: ToursDao = buildStorageDao("hotels_long_cache", longCacheTtl)
  lazy val hotelsStorage: ToursDao = {
    val dao = buildStorageDao("hotels", storageTtl)
    val instrumentedDao = new InstrumentedToursDao(metricsRegistry, dao)
    val ytDaos = ytSettings.serpCacheClusters.map(cluster => createYtDao(classOf[CacheItem], cluster))
    val serpCacheDao = new SerpCacheDao(ytDaos)
    val proxyStorage = new ProxyToursStorage(instrumentedDao, hotelsLongCache, hotelsMinPriceStorage,
      SearchType.ROOMS, calendarPushService, serpCacheDao)
    new DelayedToursDao(proxyStorage, 500.millis)
  }

  lazy val flightsStorage: FlightsDao = {
    val dao = buildFlightsDao("flights", flightsTtl)
    new ProxyFlightsDao(dao, calendarPushService)
  }

  lazy val transferStorage: TransferDao = buildTransferDao("transfers", transfersTtl)

  private def buildTransferDao(tableName: String, ttl: FiniteDuration): TransferDao = {
    lazy val ytDao = new YtTransferDao(createMirroredYtDao(classOf[Transfers], ytSettings.primaryCluster,
      ytSettings.secondaryCluster, tableName, ttl.toMillis))
    ytDao
  }

  private def buildFlightsDao(tableName: String, ttl: FiniteDuration): FlightsDao = {
    lazy val ytDao = new YtFlightsDao(createMirroredYtDao(classOf[Flights], ytSettings.primaryCluster,
      ytSettings.secondaryCluster, tableName, ttl.toMillis))
    ytDao
  }

  private def buildDirectionPriceDao(tableName: String): DirectionPriceDao = {
    lazy val ytDao = if (ytSettings.cacheDirections) {
      val inner = new YtDirectionPriceDao(createMirroredYtDao(classOf[DirectionPrice], ytSettings.primaryCluster,
        ytSettings.secondaryCluster, tableName, 1.day.toMillis))
      new LRUCacheDirectionPriceDao(inner, 150000)
    }
    else {
      new YtDirectionPriceDao(createMirroredYtDao(classOf[DirectionPrice], ytSettings.primaryCluster,
        ytSettings.secondaryCluster, tableName, 1.day.toMillis))
    }
    ytDao
  }

  private def buildMinPriceDao(table_name: String = null): MinPriceDao = {
    val yt_name = if (table_name == null) "min_price" else table_name
    lazy val ytDao = new YtMinPriceDao(
      createMirroredYtDao(classOf[Price], ytSettings.primaryCluster, ytSettings.secondaryCluster,
        yt_name, 1.day.toMillis))
    ytDao
  }

  private def buildStorageDao(name: String, ttl: FiniteDuration): ToursDao = {
    lazy val ytDao = new YtToursDao(
      createMirroredYtDao(classOf[HotelOrTour], ytSettings.primaryCluster, ytSettings.secondaryCluster, name,
        ttl.toMillis), name) with InstrumentedYtByteArrayDao
    ytDao
  }
}
