package ru.yandex.tours.storage

import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest, SearchResults}
import ru.yandex.tours.storage.yt.entities.HotelOrTour
import ru.yandex.tours.util.{GZip, Logging}
import ru.yandex.tours.util.lang.Futures._
import ru.yandex.travel.yt.YtDao
import ru.yandex.travel.yt.queries._

import scala.collection.JavaConverters._
import scala.compat.java8.FutureConverters
import scala.concurrent.{ExecutionContextExecutor, Future}


class YtToursDao(dao: YtDao[HotelOrTour], val tableName: String)
                (implicit ec: ExecutionContextExecutor) extends ByteArrayDao with Logging {
  private val CONTEXT = "CONTEXT"
  private val ALL = "ALL"

  override def saveContext(request: HotelSearchRequest, context: SearchResults.Context): Future[Unit] = {
    val obj = new HotelOrTour(request.sessionId, CONTEXT, GZip.compress(context.toByteArray))
    val started = System.currentTimeMillis()
    dao.put(obj).asScalaFuture.toUnit.andThen({
      case _ => log.debug(s"YT: Saved context in ${System.currentTimeMillis()-started} ms" )
    })
  }


  override def saveHotelSearchResult(request: HotelSearchRequest,
                                     response: SearchResults.HotelSearchResult): Future[Unit] = {
    val obj = new HotelOrTour(request.sessionId, ALL, GZip.compress(response.toByteArray))
    val started = System.currentTimeMillis()
    dao.put(obj).asScalaFuture.toUnit.andThen({
      case _ => log.debug(s"YT: Saved Hotel Search Result in ${System.currentTimeMillis()-started} ms" )
    })
  }


  override def saveOffersSearchResult(request: OfferSearchRequest,
                                      response: SearchResults.OfferSearchResult): Future[Unit] = {
    val obj = new HotelOrTour(request.hotelRequest.sessionId, request.hotelId.toString,
      GZip.compress(response.toByteArray))
    val started = System.currentTimeMillis()
    dao.put(obj).asScalaFuture.toUnit.andThen({
      case _ => log.debug(s"YT: Saved Offer Search Result in ${System.currentTimeMillis()-started} ms" )
    })
  }

  override def getContextBytes(request: HotelSearchRequest): Future[Array[Byte]] = {
    val started = System.currentTimeMillis()
    (for (res <- dao.get(request.hotelRequest.sessionId, CONTEXT).asScalaFuture) yield {
      if (!res.isPresent) {
        null
      }
      else {
        GZip.decompress(res.get().getValue)
      }
    }).andThen({case _ => log.debug(s"YT: got Context Bytes in ${System.currentTimeMillis()-started} ms" )})
  }

  override def getHotelSearchResultBytes(request: HotelSearchRequest): Future[Option[Array[Byte]]] = {
    val started = System.currentTimeMillis()
    (for (rs <- dao.get(request.sessionId, ALL).asScalaFuture) yield {
      for (item <- rs.asOption) yield GZip.decompress(item.getValue)
    }).andThen({case _ => log.debug(s"YT: got HotelSearchResult Bytes in ${System.currentTimeMillis()-started} ms" )})
  }

  override def getOffersSearchResultBytes(request: OfferSearchRequest): Future[Option[Array[Byte]]] = {
    val started = System.currentTimeMillis()
    (for (rs <- dao.get(request.hotelRequest.sessionId, request.hotelId.toString).asScalaFuture) yield {
      for (item <- rs.asOption) yield GZip.decompress(item.getValue)
    }).andThen({case _ => log.debug(s"YT: got OffersSearchResult Bytes in ${System.currentTimeMillis()-started} ms" )})
  }

  override def getOffersSearchResultsBytes(request: HotelSearchRequest,
                                           hotelIds: Iterable[Int]): Future[Seq[Array[Byte]]] = {
    val ids = hotelIds.map(t => t.toString)
    val started = System.currentTimeMillis()
    (for (rs <- FutureConverters.toScala(dao.select(Compare.eq("SessionId", request.hotelRequest.sessionId),
      new In[String]("Id", ids.asJavaCollection)))) yield {
      for {
        item <- rs.asScala
        res <- Option(GZip.decompress(item.getValue))
      } yield res
    }).andThen({case _ => log.debug(s"YT: got OffersSearchResults Bytes in ${System.currentTimeMillis()-started} ms" )})
  }
}
